<?php

/*
	Created by Mohamed.
*/

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Support\Str;
use Laravel\Sanctum\HasApiTokens;
use App\Notifications\CustomVerifyEmail;
use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Notifications\ResetPassword as ResetPasswordNotification;

class User extends Authenticatable implements MustVerifyEmail
{
	use HasApiTokens, Notifiable;

	protected $table = 'users';

	protected $casts = [
		'email_verified_at' => 'datetime',
		'is_active' => 'bool',
	];

	protected $hidden = [
		'password',
		'remember_token',
	];

	protected $fillable = [
		'full_name',
		'email',
		'email_verified_at',
		'password',
		'is_active',
		'remember_token',
		'google_id',
		'facebook_id',
		'governorate_id',
		'zip_code',
		'second_email',
		'phone',
		'image',
		'gender',
		'referral_code',
		'points',
		// 'address'
	];

	public function notes()
	{
		return $this->hasMany(Note::class, 'user_id');
	}
	public function carts()
	{
		return $this->hasMany(Cart::class);
	}

	public function credit_cards()
	{
		return $this->hasMany(CreditCard::class);
	}

	public function orders()
	{
		return $this->hasMany(Order::class);
	}

	public function reviews()
	{
		return $this->hasMany(Review::class);
	}
	public function notifications()
	{
		return $this->morphMany(Notification::class, 'notifiable');
	}

	public function addresses()
	{
		return $this->hasMany(Address::class);
	}

	// إرسال إشعار التحقق من البريد الإلكتروني
	public function sendEmailVerificationNotification()
	{
		$this->notify(new CustomVerifyEmail);
	}

	// إرسال إشعار إعادة تعيين كلمة المرور
	public function sendPasswordResetNotification($token)
	{
		$this->notify(new ResetPasswordNotification($token));
	}

	public function getEmailForVerification()
	{
		return $this->email;
	}

	public function hasVerifiedEmail()
	{
		return $this->email_verified_at !== null;
	}

	public function markEmailAsVerified()
	{
		$this->email_verified_at = Carbon::now();
		$this->save();
	}
	public function favourites()
	{
		return $this->hasMany(Favourite::class);
	}
	public function cart()
	{
		return $this->hasOne(Cart::class);
	}
	public function governorate()
	{
		return $this->belongsTo(Governorate::class, 'governorate_id');
	}

	protected static function boot()
	{
		parent::boot();

		static::creating(function ($user) {
			$user->referral_code = 'REF' . strtoupper(Str::random(8));
			$user->points = 0;
		});
	}

	// Relationships
	public function referralsMade()
	{
		return $this->hasMany(Referral::class, 'referrer_id');
	}

	public function paymants()
	{
		return $this->hasMany(Payment::class, 'user_id');
	}

	public function referredBy()
	{
		return $this->hasOne(Referral::class, 'referred_id');
	}

	public function pointsTransactions()
	{
		return $this->hasMany(PointsTransaction::class);
	}

	// Award points to the user
	public function awardPoints($points, $actionType, $relatedId = null)
	{
		$this->pointsTransactions()->create([
			'points' => $points,
			'action_type' => $actionType,
			'related_id' => $relatedId,
		]);

		$this->increment('points', $points);
	}

	// Get total referrals count
	public function getReferralCount()
	{
		return $this->referralsMade()->count();
	}
}
