<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API;

use Carbon\Carbon;
use App\Models\Banner;
use App\Models\Slider;
use App\Models\Partner;
use App\Models\Product;
use App\Models\OrderItem;
use App\Models\QuickOffer;
use Illuminate\Http\Request;
use App\Models\Advertisement;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Resources\BannerResource;
use App\Http\Resources\SliderResource;
use App\Http\Resources\PartnerResource;

use App\Http\Resources\QuickOfferResource;
use App\Http\Resources\AdvertisementResource;
use App\Http\Resources\Dashboard\ProductResource;

class ProductController extends Controller
{

    public function index(Request $request)
    {
        try {
            $productsQuery = Product::with([
                'category',
                'status',
                'cart_items',
                'discounts',
                'orderItems',
                'reviews'
            ])->where('status_id', 1);

            $search = $request->query('search', '');
            $sort = $request->query('sort', 'recent');
            $perPage = $request->query('per_page', 15);
            $page = $request->query('page', 1);
            $categoryId = $request->query('category_id', null);

            if (!empty($categoryId)) {
                $productsQuery->where('category_id', $categoryId);
            }

            if (!empty($search)) {
                $productsQuery->where(function ($query) use ($search) {
                    $query->where('name->ar', 'like', '%' . $search . '%')
                        ->orWhere('name->en', 'like', '%' . $search . '%');
                });
            }

            $columns = [
                'products.id',
                'products.category_id',
                'products.sku',
                'products.status_id',
                'products.name',
                'products.description',
                'products.short_description',
                'products.image',
                'products.price',
                'products.created_at',
                'products.updated_at',
            ];

            if ($sort === 'best_selling') {
                $salesSubquery = DB::table('order_items')
                    ->select('product_id', DB::raw('COUNT(*) as sales_count'))
                    ->groupBy('product_id');

                $productsQuery = Product::with([
                    'category',
                    'status',
                    'cart_items',
                    'discounts',
                    'orderItems',
                    'reviews'
                ])
                    ->leftJoinSub($salesSubquery, 'sales', function ($join) {
                        $join->on('products.id', '=', 'sales.product_id');
                    })
                    ->where('products.status_id', 1)
                    ->when($categoryId, function ($query, $categoryId) {
                        return $query->where('products.category_id', $categoryId);
                    })
                    ->select(
                        'products.id',
                        'products.category_id',
                        'products.sku',
                        'products.status_id',
                        'products.name',
                        'products.description',
                        'products.short_description',
                        'products.image',
                        'products.price',
                        'products.created_at',
                        'products.updated_at',
                        DB::raw('COALESCE(sales.sales_count, 0) as sales_count')
                    )
                    ->orderBy('sales_count', 'DESC')
                    ->orderBy('products.created_at', 'DESC');
            } else {
                switch ($sort) {
                    case 'high_price':
                        $productsQuery->orderBy('price', 'desc');
                        break;
                    case 'low_price':
                        $productsQuery->orderBy('price', 'asc');
                        break;
                    case 'recent':
                        $productsQuery->orderBy('created_at', 'desc');
                        break;
                    default:
                        $productsQuery->orderBy('created_at', 'desc');
                        break;
                }
                $productsQuery->select($columns);
            }

            $products = $productsQuery->paginate($perPage, ['*'], 'page', $page);

            $sliders = Slider::where('home', 0)->where('is_active', 1)->get();
            $partners = Partner::all();
            $advertisements = Advertisement::where('location_type', 'product')
                ->where('locatable_type', null)
                ->where('key', 1)
                ->get();

            return response()->json([
                'message' => 'تم جلب البيانات بنجاح',
                'products' => ProductResource::collection($products->items()),
                'advertisements' => AdvertisementResource::collection($advertisements),
                'pagination' => [
                    'current_page' => $products->currentPage(),
                    'per_page' => $products->perPage(),
                    'total' => $products->total(),
                    'last_page' => $products->lastPage(),
                    'next_page_url' => $products->nextPageUrl(),
                    'prev_page_url' => $products->previousPageUrl(),
                ],
                'sliders' => SliderResource::collection($sliders),
                'partners' => PartnerResource::collection($partners),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'حدث خطأ',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function show($id)
    {
        $product = Product::with([
            'category',
            'status',
            'cartItems',
            'discounts',
            'orderItems',
            'reviews',
            'favourites'
        ])->find($id);

        if (!$product) {
            return response()->json([
                'message' => 'Product not found'
            ], 404);
        }
        $advertisements = Advertisement::where('product_id', $id)->where('location_type', 'product')->get();
        $related_product = Product::where('category_id', $product->category_id)->get();
        $related_product_ads = Product::where('ads', 1)->where('category_id', $product->category_id)->get();


        $now = Carbon::now();
        $quick_offer = QuickOffer::where('start_date', '<=', $now)->where('end_date', '>=', $now)->with('product')->get();
        return response()->json([
            'products' => new ProductResource($product),
            'related_product' => ['product' => ProductResource::collection($related_product), 'ads' => ProductResource::collection($related_product_ads)],
            'advertisements' => AdvertisementResource::collection($advertisements),
            'quick_offer' => QuickOfferResource::collection($quick_offer),

        ], 200);
    }
}
