<?php

namespace App\Http\Controllers\API;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use App\Mail\GoogleRegistrationMail;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class GoogleAuthController extends Controller
{
    /**
     * Register a new user with Google and send a password via email
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function registerWithGoogle(Request $request)
    {
        try {
            // التحقق من صحة البيانات
            $validated = $request->validate([
                'email' => 'required|email|unique:users,email',
                'sub' => 'required|string|unique:users,google_id',
                'name' => 'required|string|max:255',
            ]);

            // إنشاء كلمة سر عشوائية واضحة
            $plainPassword = Str::random(12); // كلمة سر عشوائية بطول 12 حرفًا
            $hashedPassword = Hash::make($plainPassword);

            // استخدام المعاملات (Transaction) لإنشاء المستخدم
            $user = \Illuminate\Support\Facades\DB::transaction(function () use ($validated, $hashedPassword) {
                return User::create([
                    'full_name' => $validated['name'],
                    'email' => $validated['email'],
                    'google_id' => $validated['sub'],
                    'password' => $hashedPassword,
                ]);
            });

            // إنشاء توكن
            $token = $user->createToken('google-register')->plainTextToken;

            // إرسال بريد إلكتروني بكلمة السر
            try {
                Mail::to($user->email)->send(new GoogleRegistrationMail($plainPassword, app()->getLocale()));
            } catch (\Exception $e) {
                // تسجيل خطأ إرسال البريد
                Log::error('Failed to send Google registration email: ' . $e->getMessage(), [
                    'user_id' => $user->id,
                    'email' => $user->email,
                    'trace' => $e->getTraceAsString(),
                ]);

                // إرجاع استجابة ناجحة مع تحذير
                return response()->json([
                    'status' => 'success',
                    'message' => trans('messages.google_registration_success_email_failed'),
                    'user' => $user,
                    'token' => $token,
                ], 201);
            }

            // إرجاع استجابة ناجحة
            return response()->json([
                'status' => 'success',
                'message' => trans('messages.google_registration_success'),
                'user' => $user,
                'token' => $token,
            ], 201);
        } catch (ValidationException $e) {
            // معالجة أخطاء التحقق من البيانات
            Log::warning('Validation failed for Google registration', [
                'errors' => $e->errors(),
                'input' => $request->all(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            // معالجة أي أخطاء أخرى (مثل أخطاء قاعدة البيانات)
            Log::error('Error during Google registration: ' . $e->getMessage(), [
                'input' => $request->all(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => trans('messages.google_registration_failed'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Login a user with Google
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function loginWithGoogle(Request $request)
    {
        try {
            // التحقق من صحة البيانات
            $validated = $request->validate([
                'email' => 'required|email',
                'sub' => 'required|string',
            ]);

            // البحث عن المستخدم
            $user = User::where('email', $validated['email'])
                ->where('google_id', $validated['sub'])
                ->first();

            // التحقق من وجود المستخدم
            if (!$user) {
                Log::warning('Google login attempt for non-existent user', [
                    'email' => $validated['email'],
                    'sub' => $validated['sub'],
                ]);

                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.user_not_registered'),
                ], 401);
            }

            // إنشاء توكن
            $token = $user->createToken('google-login')->plainTextToken;

            // إرجاع استجابة ناجحة
            return response()->json([
                'status' => 'success',
                'message' => trans('messages.google_login_success'),
                'user' => $user,
                'token' => $token,
            ], 200);
        } catch (ValidationException $e) {
            // معالجة أخطاء التحقق من البيانات
            Log::warning('Validation failed for Google login', [
                'errors' => $e->errors(),
                'input' => $request->all(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            // معالجة أي أخطاء أخرى
            Log::error('Error during Google login: ' . $e->getMessage(), [
                'input' => $request->all(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status' => 'error',
                'message' => trans('messages.google_login_failed'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }
}
