<?php

/*
    Created by Mohamed.
*/

namespace App\Http\Controllers\API\Dashboard;

use DB;
use Carbon\Carbon;
use App\Models\Payment;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\PaymentResource;

class PaymentController extends Controller
{
    /**
     * Get payments with filters and pagination.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $query = Payment::query();

        $this->applyTimeFilter($query, $request->input('time_filter'));

        if ($request->has('date')) {
            $query->whereDate('created_at', $request->input('date'));
        }

        if ($request->has('invoice_no')) {
            $query->where('invoice_no', 'like', '%' . $request->input('invoice_no') . '%');
        }

        if ($request->has('type')) {
            $query->where('payment_status', $request->input('type'));
        }

        $query->with(['user', 'order']);

        $perPage = $request->input('per_page', 10);
        $payments = $query->paginate($perPage);

        $stats = $this->getPaymentStatistics($request->input('stats_filter'));

        return response()->json([
            'data' => PaymentResource::collection($payments->items()),
            'total' => $payments->total(),
            'current_page' => $payments->currentPage(),
            'last_page' => $payments->lastPage(),
            'per_page' => $payments->perPage(),
            'stats' => $stats,
        ]);
    }

    /**
     * Apply time filter to the query.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string|null $filter
     * @return void
     */
    protected function applyTimeFilter($query, $filter = null)
    {
        $now = Carbon::now();
        switch ($filter) {
            case 'day':
                $query->whereDate('created_at', $now->toDateString());
                break;
            case '7days':
                $query->where('created_at', '>=', $now->subDays(7));
                break;
            case 'month':
                $query->whereMonth('created_at', $now->month)->whereYear('created_at', $now->year);
                break;
            case '12months':
                $query->where('created_at', '>=', $now->subMonths(12));
                break;
            default:
                // No filter applied
                break;
        }
    }

    /**
     * Get payment statistics based on time filter.
     *
     * @param string|null $filter
     * @return array
     */
    protected function getPaymentStatistics($filter = null)
    {
        $query = Payment::query();
        $now = Carbon::now();

        // Apply time filter for statistics
        switch ($filter) {
            case 'day':
                $query->whereDate('created_at', $now->toDateString());
                break;
            case 'month':
                $query->whereMonth('created_at', $now->month)->whereYear('created_at', $now->year);
                break;
            case '3months':
                $query->where('created_at', '>=', $now->subMonths(3));
                break;
            case '6months':
                $query->where('created_at', '>=', $now->subMonths(6));
                break;
            default:
                // All time
                break;
        }

        $totalPayments = $query->count();
        $statusCounts = $query->select('payment_status', DB::raw('count(*) as count'))
            ->groupBy('payment_status')
            ->pluck('count', 'payment_status')
            ->all();

        return [
            'total_payments' => $totalPayments,
            'status_counts' => $statusCounts,
        ];
    }

    public function show($id)
    {
        $payment = Payment::find($id);
        return response()->json([
            'message' => 'retrieve data success',
            'data' => new PaymentResource($payment),
        ]);
    }
}