<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use App\Models\Note;
use App\Models\User;
use App\Models\Admin;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\Dashboard\UserResource;

class NoteController extends Controller
{
    public function getAllUsers()
    {
        try {
            $users = User::select('id', 'full_name', 'email')
                ->with([
                    'notes' => function ($query) {
                        $query->select('id', 'user_id', 'message', 'created_at')
                            ->latest()
                            ->take(1);
                    }
                ])
                ->get()
                ->map(function ($user) {
                    return [
                        'id' => $user->id,
                        'full_name' => $user->full_name,
                        'email' => $user->email,
                        'latest_note' => $user->notes->first() ? [
                            'id' => $user->notes->first()->id,
                            'message' => $user->notes->first()->message,
                            'created_at' => $user->notes->first()->created_at
                        ] : null
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $users
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching users: '
            ], 500);
        }
    }

    public function getUserNotes($userId)
    {
        $validator = Validator::make(['user_id' => $userId], [
            'user_id' => 'required|exists:users,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $perPage = request()->input('per_page', 10); // Default to 10 items per page
            $notes = Note::where('user_id', $userId)
                ->select('id', 'user_id', 'admin_id', 'message', 'created_at')
                ->with([
                    'admin' => function ($query) {
                        $query->select('id', 'full_name'); // Corrected to 'name' as per Admin model
                    }
                ])
                ->orderBy('created_at', 'desc')
                ->paginate($perPage);

            $formattedNotes = $notes->map(function ($note) {
                return [
                    'id' => $note->id,
                    'message' => $note->message,
                    'admin_name' => $note->admin ? $note->admin->name : 'Unknown Admin',
                    'created_at' => $note->created_at
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $formattedNotes,
                'pagination' => [
                    'current_page' => $notes->currentPage(),
                    'per_page' => $notes->perPage(),
                    'total' => $notes->total(),
                    'last_page' => $notes->lastPage(),
                    'next_page_url' => $notes->nextPageUrl(),
                    'prev_page_url' => $notes->previousPageUrl()
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching notes: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateNote(Request $request, $userId, $noteId)
    {


        try {
            $validator = Validator::make(array_merge($request->all(), ['note_id' => $noteId]), [
                'note_id' => 'required|exists:notes,id',
                'message' => 'required|string|min:1'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }
            $adminId = User::find($userId)->id;

            $note = Note::findOrFail($noteId);

            // Optional: Ensure the note belongs to the authenticated admin
            if ($note->user_id !== $adminId) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized to update this note'
                ], 403);
            }

            $note->update([
                'message' => $request->message,
                'updated_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Note updated successfully',
                'data' => [
                    'id' => $note->id,
                    'user_id' => $note->user_id,
                    'admin_id' => $note->admin_id,
                    'message' => $note->message,
                    'created_at' => $note->created_at,
                    'updated_at' => $note->updated_at
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating note: ' . $e->getMessage()
            ], 500);
        }
    }

    public function createNotes(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
            'message' => 'required|string|min:1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $adminId = auth('admin')->id(); // Use admin guard

            // Validate that the authenticated admin exists
            if (!Admin::find($adminId)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authenticated admin does not exist'
                ], 403);
            }

            $notes = [];
            foreach ($request->user_ids as $userId) {
                $notes[] = Note::create([
                    'user_id' => $userId,
                    'admin_id' => $adminId,
                    'message' => $request->message
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Notes created successfully',
                'data' => $notes
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating notes: ' . $e->getMessage()
            ], 500);
        }
    }
}