<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\Discount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DiscountController extends Controller
{
    // Get all discounts without pagination
    public function getDiscounts(Request $request)
    {
        try {
            $discounts = Discount::select('id', 'product_id', 'value', 'type', 'place', 'usage_limit', 'used_count', 'start_date', 'end_date', 'created_at')
                ->with('product:id,name') // Eager load product name for context
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $discounts->map(function ($discount) {
                    $decodedName = is_string($discount->product->name)
                        ? json_decode($discount->product->name, true)
                        : $discount->name;
                    $decodedName2 = is_string($discount->productIn?->name)
                        ? json_decode($discount->productIn?->name, true)
                        : $discount->name;
                    return [
                        'id' => $discount->id,
                        'product_id' => $discount->product_id,
                        'product_name' => [
                            'ar' => $decodedName['ar'] ?? null,
                            'en' => $decodedName['en'] ?? null,
                        ],
                        'product_name_in_place' => [
                            'ar' => $decodedName2['ar'] ?? null,
                            'en' => $decodedName2['en'] ?? null,
                        ],
                        'value' => $discount->value,

                        'type' => $discount->type,
                        'place' => $discount->place,
                        'usage_limit' => $discount->usage_limit,
                        'used_count' => $discount->used_count,
                        'start_date' => $discount->start_date,
                        'end_date' => $discount->end_date,
                        'created_at' => $discount->created_at
                    ];
                })
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching discounts: ' . $e->getMessage()
            ], 500);
        }
    }

    // Create a new discount
    public function createDiscount(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_id' => 'nullable|exists:products,id',
            'value' => 'required|numeric|min:0',
            'type' => 'required|in:percentage,fixed',
            'place' => 'nullable|max:255',
            'usage_limit' => 'nullable|integer|min:1',
            'used_count' => 'nullable|integer|min:0',
            'start_date' => 'nullable',
            'end_date' => 'nullable|after_or_equal:start_date'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $adminId = auth('admin')->id();
            if (!\App\Models\Admin::find($adminId)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authenticated admin does not exist'
                ], 403);
            }

            $discount = Discount::where('product_id', $request->product_id)
                ->where('place', $request->place)
                ->first();

            if ($discount) {
                $discount->update([
                    'value' => $request->value,
                    'type' => $request->type,
                    'usage_limit' => $request->usage_limit,
                    'used_count' => $request->input('used_count', 0),
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'ads' => $request->ads ?? 0

                ]);
            } else {
                $discount = Discount::create([
                    'product_id' => $request->product_id,
                    'value' => $request->value,
                    'type' => $request->type,
                    'place' => $request->place,
                    'usage_limit' => $request->usage_limit,
                    'used_count' => $request->input('used_count', 0),
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'ads' => $request->ads ?? 0,

                ]);
            }

            return response()->json([
                'success' => true,
                'message' => $discount->wasRecentlyCreated
                    ? 'Discount created successfully'
                    : 'Discount updated successfully',
                'data' => [
                    'id' => $discount->id,
                    'product_id' => $discount->product_id,
                    'product_name' => $discount->product ? $discount->product->name : null,
                    'product_name_in_place' => $discount->productIn ? $discount->productIn->name : null,
                    'value' => $discount->value,
                    'type' => $discount->type,
                    'place' => $discount->place,
                    'ads' => $discount->ads,
                    'usage_limit' => $discount->usage_limit,
                    'used_count' => $discount->used_count,
                    'start_date' => $discount->start_date,
                    'end_date' => $discount->end_date,
                    'created_at' => $discount->created_at
                ]
            ], $discount->wasRecentlyCreated ? 201 : 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating/updating discount: ' . $e->getMessage()
            ], 500);
        }
    }


    // Delete a discount
    public function deleteDiscount($id)
    {
        $validator = Validator::make(['id' => $id], [
            'id' => 'required|exists:discounts,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $adminId = auth('admin')->id();
            if (!\App\Models\Admin::find($adminId)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Authenticated admin does not exist'
                ], 403);
            }

            $discount = Discount::findOrFail($id);
            $discount->delete();

            return response()->json([
                'success' => true,
                'message' => 'Discount deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting discount: ' . $e->getMessage()
            ], 500);
        }
    }
}
