<?php

/*
    Created by Mohamed.
*/

namespace App\Http\Controllers\API\Dashboard;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Order;
use App\Models\Point;
use App\Models\Status;
use App\Models\Payment;
use App\Exports\UsersExport;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Resources\OrderResource;
use App\Http\Resources\StatusResource;
use App\Http\Resources\PaymentResource;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\NotificationResource;

class ClientsController extends Controller
{
    public function index(Request $request)
    {
        $query = User::query();

        // ====== 1. Search by name or email ======
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('full_name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // ====== 2. Filter by exact date ======
        if ($request->has('date') && $request->date) {
            $date = Carbon::parse($request->date)->format('Y-m-d');
            $query->whereDate('created_at', $date);
        }

        // ====== 3. Filter by period ======
        if ($request->has('period') && $request->period) {
            $period = $request->period;
            $query->where('created_at', '>=', match ($period) {
                '24_hours' => Carbon::now()->subHours(24),
                '7_days' => Carbon::now()->subDays(7),
                '30_days' => Carbon::now()->subDays(30),
                '12_months' => Carbon::now()->subMonths(12),
                default => Carbon::now()->subYears(100),
            });
        }

        // ====== 4. Pagination ======
        $perPage = $request->get('per_page', 15); // default to 15
        $users = $query->paginate($perPage);

        // ====== 5. Count stats (active/inactive/total) ======
        $countPeriod = $request->get('count_period', 'all_time');
        $startDate = match ($countPeriod) {
            '24_hours' => Carbon::now()->subHours(24),
            '30_days' => Carbon::now()->subDays(30),
            '3_months' => Carbon::now()->subMonths(3),
            '6_months' => Carbon::now()->subMonths(6),
            default => Carbon::now()->subYears(100),
        };

        $count_active_users = User::where('created_at', '>=', $startDate)
            ->where(function ($q) {
                $q->whereNotNull('email_verified_at')
                    ->orWhere('is_active', 1);
            })->count();

        $count_inactive_users = User::where('created_at', '>=', $startDate)
            ->where(function ($q) {
                $q->whereNull('email_verified_at')
                    ->orWhere('is_active', 0);
            })->count();

        $count_all_users = User::where('created_at', '>=', $startDate)->count();

        return response()->json([
            'message' => 'تم استرجاع البيانات بنجاح',
            'users' => UserResource::collection($users),
            'pagination' => [
                'current_page' => $users->currentPage(),
                'last_page' => $users->lastPage(),
                'per_page' => $users->perPage(),
                'total' => $users->total(),
            ],
            'count_active_users' => $count_active_users,
            'count_inactive_users' => $count_inactive_users,
            'count_all_users' => $count_all_users,
        ], 200);
    }


    public function export(Request $request)
    {
        return Excel::download(new UsersExport($request), 'clients_' . Carbon::now()->format('Y-m-d_H-i-s') . '.xlsx');
    }

    public function show($id, Request $request)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'المستخدم غير موجود',
            ], 404);
        }

        $date = $request->input('date');
        $statusId = $request->input('status_id');

        $orderQuery = Order::where('user_id', $id);
        $pointQuery = Point::where('model_id', $id);
        $notifications = Notification::where('notifiable_id', $user->id)->Where('notifiable_type', User::class)->get();

        if ($date) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy',
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $orderQuery->whereDate('created_at', $formattedDate);
            $pointQuery->whereDate('created_at', $formattedDate);
        }

        if ($statusId) {
            $validator = Validator::make(['status_id' => $statusId], [
                'status_id' => 'integer|exists:statuses,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم حالة الطلب غير صحيح أو غير موجود',
                ], 400);
            }

            $orderQuery->where('status_id', $statusId);
        }

        return response()->json([
            'message' => 'تم استرجاع بيانات المستخدم بنجاح',
            'user' => new UserResource($user),
            'orders' => OrderResource::collection($orderQuery->get()),
            'notifictions' => NotificationResource::collection($notifications),
            'count_order' => $orderQuery->count(),
            'points_count' => $pointQuery->count('point_x'),
            'points_value' => $pointQuery->count('point_y'),
            'statuses' => StatusResource::collection(
                Status::whereBetween('id', [1, 4])->get()
            ),
        ], 200);
    }

    public function update(Request $request, $id)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'المستخدم غير موجود',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'full_name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|max:255|unique:users,email,' . $id,
            'password' => 'sometimes|string|min:8|confirmed',
            'is_active' => 'sometimes|boolean',
            'governorate_id' => 'sometimes|nullable|string|max:255',
            'zip_code' => 'sometimes|nullable|string|max:255',
            'second_email' => 'sometimes|nullable|email|max:255|unique:users,second_email,' . $id,
            'phone' => 'sometimes|nullable|string|max:255',
            'image' => 'sometimes|nullable|string|max:255',
            'status' => 'sometimes|in:active,block,inactive,on-process',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'خطأ في التحقق من البيانات',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'full_name',
            'email',
            'password',
            'is_active',
            'governorate_id',
            'zip_code',
            'second_email',
            'phone',
            'image',
            'status',
        ]);

        if ($request->has('password')) {
            $data['password'] = bcrypt($request->password);
        }

        $user->update($data);

        return response()->json([
            'message' => 'تم تحديث بيانات المستخدم بنجاح',
            'user' => new UserResource($user),
        ], 200);
    }

    public function updateStatus(Request $request, $id)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'المستخدم غير موجود',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'status' => 'required|in:active,block,inactive,on-process',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'خطأ في التحقق من البيانات',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user->update(['status' => $request->status]);

        return response()->json([
            'message' => 'تم تحديث حالة المستخدم بنجاح',
            'user' => new UserResource($user),
        ], 200);
    }

    public function destroy($id)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'المستخدم غير موجود',
            ], 404);
        }

        $user->delete();

        return response()->json([
            'message' => 'تم حذف المستخدم بنجاح',
        ], 200);
    }
    public function notices($id, Request $request)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'المستخدم غير موجود',
            ], 404);
        }

        $date = $request->input('date');
        $status = $request->input('status');
        $perPage = $request->input('per_page', 10); // عدد العناصر لكل صفحة، افتراضي 10

        $orderQuery = Order::where('user_id', $id);
        $pointQuery = Point::where('model_id', $id);
        $notificationsQuery = Notification::where('notifiable_id', $user->id)
            ->where('notifiable_type', User::class);

        // التحقق من صحة التاريخ إذا تم تمريره
        if ($date) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy',
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $orderQuery->whereDate('created_at', $formattedDate);
            $pointQuery->whereDate('created_at', $formattedDate);
            $notificationsQuery->whereDate('created_at', $formattedDate);
        }

        // التحقق من صحة الحالة إذا تم تمريرها
        if ($status) {
            $validator = Validator::make(['status' => $status], [
                'status' => 'in:recieved,not-recieved'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'حالة الإشعار غير صحيحة، استخدم recieved أو not-recieved',
                ], 400);
            }

            $notificationsQuery->where('status', $status);
        }

        // تنفيذ الاستعلام مع الـ pagination
        $notifications = $notificationsQuery->paginate($perPage);

        return response()->json([
            'message' => 'تم استرجاع بيانات المستخدم بنجاح',
            //  'user' => new UserResource($user),
            'notifications' => NotificationResource::collection($notifications),
            'pagination' => [
                'current_page' => $notifications->currentPage(),
                'per_page' => $notifications->perPage(),
                'total' => $notifications->total(),
                'last_page' => $notifications->lastPage(),
                'from' => $notifications->firstItem(),
                'to' => $notifications->lastItem(),
            ],
            'count_order' => $orderQuery->count(),
            'points_count' => $pointQuery->count('point_x'),
            'points_value' => $pointQuery->count('point_y'),
        ], 200);
    }
    public function payments($id, Request $request)
    {
        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'المستخدم غير موجود',
            ], 404);
        }
        $invoiceNo = $request->input('invoice_no');
        $date = $request->input('date');
        $status = $request->input('payment_status');
        $perPage = $request->input('per_page', 10);

        $paymentsQuery = Payment::where('user_id', $id);

        // التحقق من رقم الفاتورة إذا تم تمريره
        if ($invoiceNo) {
            $paymentsQuery->where('invoice_no', 'like', '%' . $invoiceNo . '%');
        }

        if ($date) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy',
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $paymentsQuery->whereDate('created_at', $formattedDate);
        }

        if ($status) {

            $paymentsQuery->where('payment_status', $status);
        }

        $payments = $paymentsQuery->paginate($perPage);

        return response()->json([
            'message' => 'تم استرجاع بيانات المدفوعات بنجاح',
            'payments' => PaymentResource::collection($payments),
            'pagination' => [
                'current_page' => $payments->currentPage(),
                'per_page' => $payments->perPage(),
                'total' => $payments->total(),
                'last_page' => $payments->lastPage(),
                'from' => $payments->firstItem(),
                'to' => $payments->lastItem(),
            ],
        ], 200);
    }
}