<?php

/*
    Created by Mohamed.
*/

namespace App\Http\Controllers\API\Dashboard;

use App\Models\Category;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use App\Http\Resources\CategoryResource;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\Dashboard\CategoryDetailsResource;

class CategoryController extends Controller
{
    /**
     * Retrieve all categories
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        $mainCategories = Category::whereNull('parent_id')->get();

        $subCategories = Category::whereNotNull('parent_id')->get();

        $categories = $mainCategories->merge($subCategories);

        return response()->json([
            'status' => 'success',
            'data'   => CategoryDetailsResource::collection($categories)
        ], 200);
    }

    /**
     * Create a new category with multilingual support (ar, en)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|array',
            'name.ar' => 'required|string|max:255',
            'name.en' => 'required|string|max:255',

            'parent_id' => 'nullable|exists:categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // Max 2MB

            'description' => 'nullable|array',
            'description.ar' => 'nullable|string',
            'description.en' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->all();

        // Encode multilingual fields
        $data['name'] = json_encode($request->name, JSON_UNESCAPED_UNICODE);
        if ($request->has('description')) {
            $data['description'] = json_encode($request->description, JSON_UNESCAPED_UNICODE);
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('categories', 'public');
            $data['image'] = Storage::url($imagePath);
        }

        $category = Category::create($data);

        return response()->json([
            'status' => 'success',
            'data' => new CategoryResource($category)
        ], 201);
    }

    /**
     * Update an existing category with multilingual support (ar, en)
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $category = Category::find($id);

        if (!$category) {
            return response()->json([
                'status' => 'error',
                'message' => 'Category not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|array',
            'name.ar' => 'sometimes|required|string|max:255',
            'name.en' => 'sometimes|required|string|max:255',

            'parent_id' => 'nullable|exists:categories,id',
            'image' => 'nullable',

            'description' => 'nullable|array',
            'description.ar' => 'nullable|string',
            'description.en' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->except('image'); // استبعد الصورة مؤقتاً

        // Encode multilingual fields
        if ($request->has('name')) {
            $data['name'] = json_encode($request->name, JSON_UNESCAPED_UNICODE);
        }
        if ($request->has('description')) {
            $data['description'] = json_encode($request->description, JSON_UNESCAPED_UNICODE);
        }

        // تحديث صورة التصنيف
        if ($request->hasFile('image') && $request->file('image')->isValid()) {
            if ($category->image && Storage::disk('public')->exists($category->image)) {
                Storage::disk('public')->delete($category->image);
            }

            $imagePath = $request->file('image')->store('categories', 'public');
            $data['image'] = Storage::url($imagePath);
        } elseif ($request->has('image') && ($request->image === null || $request->image === 'null' || $request->image === '')) {
            if ($category->image && Storage::disk('public')->exists($category->image)) {
                Storage::disk('public')->delete($category->image);
            }
            $data['image'] = null;
        }

        // تحديث البيانات كلها مرة واحدة
        $category->update($data);

        return response()->json([
            'status' => 'success',
            'data' => new CategoryResource($category)
        ], 200);
    }


    /**
     * Delete a category
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $category = Category::find($id);

        if (!$category) {
            return response()->json([
                'status' => 'error',
                'message' => 'Category not found'
            ], 404);
        }

        // Check if category has children or products
        if ($category->categories()->exists() || $category->products()->exists()) {
            return response()->json([
                'status' => 'error',
                'message' => 'Cannot delete category with associated subcategories or products'
            ], 422);
        }

        // Delete image from storage if exists
        if ($category->image) {
            $imagePath = str_replace(Storage::url(''), '', $category->image);
            Storage::disk('public')->delete($imagePath);
        }

        $category->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Category deleted successfully'
        ], 200);
    }
}
