<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use Illuminate\Http\Request;
use App\Models\Advertisement;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\Dashboard\AdvertisementResource;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AdvertisementController extends Controller
{
    /**
     * Retrieve primary and secondary advertisements.
     *
     * @return JsonResponse
     */
    public function index(Request $request)
    {
        if ($request->type == 'home') {
            $ads = Advertisement::where('location_type', 'home')->where('key', $request->key)->get();
        } else {
            $ads = Advertisement::where('location_type', 'product')->get();
        }

        return response()->json([
            'message' => 'Advertisements retrieved successfully',
            'ads' =>  AdvertisementResource::collection($ads),

        ], 200);
    }
    /**
     * Create a new advertisement.
     *
     * @param Request $request
     * @return JsonResponse
     */

    public function upsert(Request $request, $id = null)
    {
        try {
            // Define validation rules
            $rules = [
                'title' => [$id ? 'sometimes' : 'required', 'array'],
                'title.en' => [$id ? 'sometimes' : 'required', 'string', 'max:255'],
                'title.ar' => [$id ? 'sometimes' : 'required', 'string', 'max:255'],
                'description' => 'nullable|array',
                'description.en' => 'nullable|string',
                'description.ar' => 'nullable|string',
                'image' => 'nullable',
                'key' => [$id ? 'sometimes' : 'required', 'boolean'],
                'link' => [$id ? 'sometimes' : 'required', 'string'],
                'status' => 'nullable|string',
                'default_image' => 'nullable',
                'product_id' => 'nullable|integer|exists:products,id',
                'color' => [
                    $id ? 'sometimes' : 'required',
                    'string',
                    'max:255',
                    'regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/'
                ],
                'type' => [$id ? 'sometimes' : 'required', 'in:home,product'],
                'locatable_id' => 'nullable|integer|min:0',
                'order' => 'nullable|integer|min:0',
                'price' => 'nullable|numeric|min:0',
                'discount' => 'nullable|numeric|min:0|max:100',
                'btn_bg_color' => 'nullable',
                'border_btn_color' => 'nullable',
                'text_btn' => 'nullable',
                'bg_ads_color_default' => 'nullable',
            ];

            // Define custom validation messages
            $messages = [
                'title.required' => 'عنوان الإعلان مطلوب',
                'title.array' => 'عنوان الإعلان يجب أن يكون كائن JSON',
                'title.en.required' => 'عنوان الإعلان بالإنجليزية مطلوب',
                'title.en.string' => 'عنوان الإعلان بالإنجليزية يجب أن يكون نصًا',
                'title.ar.required' => 'عنوان الإعلان بالعربية مطلوب',
                'title.ar.string' => 'عنوان الإعلان بالعربية يجب أن يكون نصًا',
                'description.array' => 'وصف الإعلان يجب أن يكون كائن JSON',
                'image.required' => 'صورة الإعلان مطلوبة',
                'image.image' => 'يجب أن تكون الصورة من نوع صورة صالحة',
                'image.mimes' => 'يجب أن تكون الصورة بصيغة: jpg أو png أو jpeg',
                'image.max' => 'أقصى حجم مسموح للصورة هو 2 ميجابايت',
                'key.required' => 'حقل التمييز مطلوب',
                'key.boolean' => 'قيمة التمييز يجب أن تكون منطقية',
                'link.required' => 'الرابط مطلوب',
                'link.string' => 'الرابط يجب أن يكون نصًا',
                'status.string' => 'حالة الإعلان يجب أن تكون نصًا',
                'product_id.integer' => 'رقم المنتج يجب أن يكون رقمًا صحيحًا',
                'product_id.exists' => 'المنتج غير موجود',
                'color.required' => 'لون الخلفية مطلوب',
                'color.regex' => 'صيغة اللون غير صحيحة، يجب أن تكون مثل: #ffffff',
                'type.required' => 'نوع الإعلان مطلوب',
                'type.in' => 'نوع الإعلان يجب أن يكون home أو product',
                'locatable_id.integer' => 'معرّف العنصر يجب أن يكون رقمًا صحيحًا',
                'order.integer' => 'ترتيب الإعلان يجب أن يكون رقمًا صحيحًا',
                'order.min' => 'ترتيب الإعلان يجب ألا يكون أقل من 0',
            ];

            // Validate the request
            $validator = Validator::make($request->all(), $rules, $messages);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'فشل التحقق من البيانات',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $advertisement = $id ? Advertisement::find($id) : new Advertisement();

            if ($id && !$advertisement) {
                return response()->json([
                    'status' => false,
                    'message' => 'الإعلان غير موجود',
                ], 404);
            }

            $imagePath = $advertisement->image ?? null;

            if ($request->hasFile('image') && $request->file('image')->isValid()) {
                if ($id && $imagePath && Storage::disk('public')->exists($imagePath)) {
                    Storage::disk('public')->delete($imagePath);
                }
                $imagePath = $request->file('image')->store('advertisements', 'public');
            } elseif ($request->has('image')) {
                if ($request->image === null || $request->image === 'null' || $request->image === '') {
                    if ($imagePath && Storage::disk('public')->exists($imagePath)) {
                        Storage::disk('public')->delete($imagePath);
                    }
                    $imagePath = null;
                } elseif ($request->image === Storage::url($imagePath)) {
                    $imagePath = $advertisement->image;
                }
            }

            $defaultImagePath = $advertisement->default_image ?? null;

            if ($request->hasFile('default_image') && $request->file('default_image')->isValid()) {
                if ($id && $defaultImagePath && Storage::disk('public')->exists($defaultImagePath)) {
                    Storage::disk('public')->delete($defaultImagePath);
                }
                $defaultImagePath = $request->file('default_image')->store('advertisements', 'public');
            } elseif ($request->has('default_image')) {
                if ($request->default_image === null || $request->default_image === 'null' || $request->default_image === '') {
                    if ($defaultImagePath && Storage::disk('public')->exists($defaultImagePath)) {
                        Storage::disk('public')->delete($defaultImagePath);
                    }
                    $defaultImagePath = null;
                } elseif ($request->default_image === Storage::url($defaultImagePath)) {
                    $defaultImagePath = $advertisement->default_image;
                }
            }


            $locatableType = $request->has('type')
                ? ($request->type === 'product' ? 'App\\Models\\Product' : null)
                : ($advertisement->locatable_type ?? null);
            $locationType = $request->has('type') ? $request->type : ($advertisement->location_type ?? null);

            $title = $request->has('title') ? json_encode($request->title) : ($advertisement->title ?? null);
            $description = $request->has('description')
                ? ($request->description ? json_encode($request->description) : null)
                : ($advertisement->description ?? null);

            $data = [
                'title'          => $title,
                'description'    => $description,
                'image'          => $imagePath,
                'default_image'   => $defaultImagePath,
                'location_type'  => $locationType,
                'locatable_id'   => $request->locatable_id ?? ($advertisement->locatable_id ?? null),
                'locatable_type' => $locatableType,
                'key'            => $request->has('key') ? $request->key : ($advertisement->key ?? false),
                'link'           => $request->link ?? ($advertisement->link ?? null),
                'status'         => $request->status ?? ($advertisement->status ?? null),
                'product_id'     => $request->product_id ?? ($advertisement->product_id ?? null),
                'color'          => $request->color ?? ($advertisement->color ?? '#ffca1a'),
                'order'          => $request->order ?? ($advertisement->order ?? null),
                'price' => $request->price ?? ($advertisement->price ?? null),
                'discount' => $request->discount ?? ($advertisement->discount ?? null),
                'btn_bg_color' => $request->btn_bg_color ?? ($advertisement->btn_bg_color ?? null),
                'border_btn_color' => $request->border_btn_color ?? ($advertisement->border_btn_color ?? null),
                'text_btn' => $request->text_btn ?? ($advertisement->text_btn ?? null),
                'bg_ads_color_default' => $request->bg_ads_color_default ?? ($advertisement->bg_ads_color_default ?? null),
            ];


            // Create or update the advertisement
            if ($id) {
                $advertisement->update($data);
            } else {
                $advertisement = Advertisement::create($data);
            }

            $imageUrl = $imagePath ? Storage::url($imagePath) : null;

            $advertisement->title = json_decode($advertisement->title, true);
            $advertisement->description = $advertisement->description ? json_decode($advertisement->description, true) : null;

            return response()->json([
                'status' => true,
                'message' => $id ? 'تم تعديل الإعلان بنجاح' : 'تم إنشاء الإعلان بنجاح',
                'data' => new AdvertisementResource($advertisement),
            ], $id ? 200 : 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'حدث خطأ غير متوقع أثناء ' . ($id ? 'تعديل' : 'إنشاء') . ' الإعلان',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    /**
     * Summary of destroy
     * @param mixed $id
     * @return JsonResponse
     */
    public function destroy($id): JsonResponse
    {
        try {
            // Find the advertisement
            $advertisement = Advertisement::find($id);

            if (!$advertisement) {
                return response()->json([
                    'status' => false,
                    'message' => 'الإعلان غير موجود',
                ], 404);
            }

            if ($advertisement->image && Storage::disk('public')->exists($advertisement->image)) {
                Storage::disk('public')->delete($advertisement->image);
            }

            $advertisement->delete();

            return response()->json([
                'status' => true,
                'message' => 'تم حذف الإعلان بنجاح',
                'data' => null,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'حدث خطأ غير متوقع أثناء حذف الإعلان',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
