<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use Exception;
use App\Models\Role;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use App\Http\Resources\AdminResource;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AdminController extends Controller
{
    // Get all admins
    public function index(): JsonResponse
    {
        $admins = Admin::with('roles')->get();
        return response()->json(AdminResource::collection($admins));
    }

    // Get a single admin
    public function show(Admin $admin): JsonResponse
    {
        $admin->load('roles');
        return response()->json(new AdminResource($admin));
    }

    // Create an admin



    public function store(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:admins,email',
                'phone_number' => 'required|string|max:15|unique:admins,phone_number',
                'password' => 'required|string|min:8',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            ], [
                'name.required' => 'الاسم مطلوب.',
                'email.required' => 'البريد الإلكتروني مطلوب.',
                'email.email' => 'صيغة البريد الإلكتروني غير صحيحة.',
                'email.unique' => 'البريد الإلكتروني مستخدم من قبل.',
                'phone_number.required' => 'رقم الهاتف مطلوب.',
                'phone_number.unique' => 'رقم الهاتف مستخدم من قبل.',
                'password.required' => 'كلمة المرور مطلوبة.',
                'password.min' => 'كلمة المرور يجب أن تكون 8 أحرف على الأقل.',
                'image.image' => 'الملف يجب أن يكون صورة.',
                'image.mimes' => 'الصورة يجب أن تكون بصيغة: jpeg, png, jpg, gif.',
                'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميجابايت.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'يوجد خطأ في البيانات المدخلة.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $validator->validated();

            if ($request->hasFile('image')) {
                $data['image'] = $request->file('image')->store('admin_images', 'public');
            }

            $data['password'] = Hash::make($data['password']);

            $admin = Admin::create($data);

            return response()->json([
                'message' => 'تم إنشاء المشرف بنجاح.',
                'data' => new AdminResource($admin)
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'حدث خطأ غير متوقع أثناء إنشاء المشرف.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    // Update admin data and roles (for any admin)
    public function update(Request $request, Admin $admin): JsonResponse
    {
        $validated = $request->validate([
            'type' => 'required|in:super,editor',
            'name' => 'required|string|max:255',
            'phone_number' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255|unique:admins,email,' . $admin->id,
            'password' => 'nullable|string|min:8',
            'gender' => 'required|in:male,female',
            'code' => 'nullable',
            'role_ids' => 'nullable|array',
            'role_ids.*' => 'exists:roles,id',
        ]);

        if (isset($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        } else {
            unset($validated['password']);
        }

        $admin->update($validated);

        // Update roles if provided
        if (isset($validated['role_ids'])) {
            $admin->roles()->sync($validated['role_ids']);
        }

        return response()->json(new AdminResource($admin->load('roles')));
    }

    // Update authenticated admin's data and roles

    public function updateUser(Request $request): JsonResponse
    {
        try {
            $admin = auth()->user();

            if (!$admin) {
                return response()->json(['message' => 'غير مصرح'], 401);
            }

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'phone_number' => 'nullable|string|max:20',
                'email' => 'nullable|email|max:255|unique:admins,email,' . $admin->id,
                'gender' => 'required|in:male,female',
                'city' => 'nullable|string|max:255',
                'street' => 'nullable|string|max:255',
                'addres_line' => 'nullable|string|max:255',
                'qualification' => 'nullable|string|max:255',
                'current_adjective' => 'nullable|string|max:255',
                'country_id' => 'nullable|exists:countries,id',
            ], [
                'name.required' => 'الاسم مطلوب.',
                'name.string' => 'الاسم يجب أن يكون نصاً.',
                'name.max' => 'الاسم لا يجب أن يتجاوز 255 حرفًا.',
                'phone_number.string' => 'رقم الهاتف يجب أن يكون نصاً.',
                'phone_number.max' => 'رقم الهاتف لا يجب أن يتجاوز 20 رقمًا.',
                'email.email' => 'صيغة البريد الإلكتروني غير صحيحة.',
                'email.max' => 'البريد الإلكتروني لا يجب أن يتجاوز 255 حرفًا.',
                'email.unique' => 'البريد الإلكتروني مستخدم بالفعل.',
                'gender.required' => 'النوع مطلوب.',
                'gender.in' => 'النوع يجب أن يكون male أو female.',

                'country_id.exists' => 'الدولة المحددة غير موجودة.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'فشل التحقق من البيانات.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $validated = $validator->validated();

            if (isset($validated['password'])) {
                $validated['password'] = Hash::make($validated['password']);
            } else {
                unset($validated['password']);
            }

            $admin->update($validated);

            if (isset($validated['role_ids'])) {
                $admin->roles()->sync($validated['role_ids']);
            }

            return response()->json([
                'status' => true,
                'message' => 'تم تحديث البيانات بنجاح.',
                'data' => new AdminResource($admin->load('roles', 'country'))
            ], 200);

        } catch (Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'حدث خطأ أثناء تحديث البيانات.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // Update admin profile image
    public function updateImage(Request $request, Admin $admin): JsonResponse
    {
        $validated = $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        // Delete old image if exists
        if ($admin->image) {
            Storage::disk('public')->delete($admin->image);
        }

        $imagePath = $request->file('image')->store('admin_images', 'public');
        $admin->update(['image' => $imagePath]);

        return response()->json(new AdminResource($admin));
    }

    // Delete admin profile image
    public function deleteImage(Admin $admin): JsonResponse
    {
        if ($admin->image) {
            Storage::disk('public')->delete($admin->image);
            $admin->update(['image' => null]);
        }

        return response()->json(new AdminResource($admin));
    }

    // Delete an admin
    public function destroy(Admin $admin): JsonResponse
    {
        // Delete associated image
        if ($admin->image) {
            Storage::disk('public')->delete($admin->image);
        }

        $admin->delete();
        return response()->json(null, 204);
    }

    // Assign a role to an admin
    public function assignRole(Request $request, Admin $admin): JsonResponse
    {
        $validated = $request->validate([
            'role_id' => 'required|exists:roles,id',
        ]);

        $admin->roles()->attach($validated['role_id']);
        return response()->json(new AdminResource($admin->roles), 201);
    }

    // Get authenticated admin
    public function getUser(): JsonResponse
    {
        return response()->json([
            'message' => 'get data success',
            'data' => new AdminResource(auth()->user())
        ], 200);
    }

    public function UpdateImageProfile(Request $request): JsonResponse
    {
        $admin = auth()->user();

        if (!$admin) {
            return response()->json(['message' => 'Unauthenticated'], 401);
        }

        $request->validate([
            'image' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048',
        ], [
            'image.required' => 'الصورة مطلوبة.',
            'image.image' => 'الملف يجب أن يكون صورة.',
            'image.mimes' => 'الصورة يجب أن تكون بصيغة: jpg, jpeg, png, webp.',
            'image.max' => 'أقصى حجم للصورة هو 2 ميجا.',
        ]);

        // حذف الصورة القديمة إن وجدت
        if ($admin->image && Storage::disk('public')->exists($admin->image)) {
            Storage::disk('public')->delete($admin->image);
        }

        // رفع الصورة الجديدة
        $path = $request->file('image')->store('admins', 'public');

        $admin->image = $path;
        $admin->save();

        return response()->json([
            'message' => 'تم تحديث صورة البروفايل بنجاح.',
            'image_url' => url('storage/' . $path),
        ]);
    }

    public function DeleteImageProfile(): JsonResponse
    {
        $admin = auth()->user();

        if (!$admin) {
            return response()->json(['message' => 'Unauthenticated'], 401);
        }

        if (!$admin->image) {
            return response()->json(['message' => 'لا توجد صورة حالية للحذف.'], 404);
        }

        // حذف الصورة من التخزين
        if (Storage::disk('public')->exists($admin->image)) {
            Storage::disk('public')->delete($admin->image);
        }

        $admin->image = null;
        $admin->save();

        return response()->json(['message' => 'تم حذف صورة البروفايل.']);
    }

}