<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\CreditCard;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CreditCardController extends Controller
{
    /**
     * Get all credit cards for a specific user.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        // Get the authenticated user's ID
        $userId = $request->user()->id;

        // Fetch all credit cards for the user
        $creditCards = CreditCard::where('user_id', $userId)->get();

        return response()->json([
            'success' => true,
            'data' => $creditCards
        ], 200);
    }

    /**
     * Add a new credit card.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        // Validate the request data
        $validator = Validator::make($request->all(), [
            'card_number_encrypted' => 'required|string',
            'cardholder_name' => 'required|string',
            'expiry_date' => 'required|date_format:m/y',
            'last_four_digits' => 'required|string|min:4|max:4',
            'is_default' => 'boolean',
            'image' => 'nullable|string',
            'card_type' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Create a new credit card
        $creditCard = CreditCard::create([
            'user_id' => $request->user()->id,
            'card_number_encrypted' => $request->input('card_number_encrypted'),
            'cardholder_name' => $request->input('cardholder_name'),
            'expiry_date' => $request->input('expiry_date'),
            'last_four_digits' => $request->input('last_four_digits'),
            'is_default' => $request->input('is_default', false),
            'image' => $request->input('image'),
            'card_type' => $request->input('card_type'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Credit card added successfully',
            'data' => $creditCard
        ], 201);
    }

    /**
     * Get a specific credit card by ID.
     *
     * @param int $id
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id, Request $request)
    {
        // Get the authenticated user's ID
        $userId = $request->user()->id;

        // Find the credit card by ID and ensure it belongs to the user
        $creditCard = CreditCard::where('id', $id)->where('user_id', $userId)->first();

        if (!$creditCard) {
            return response()->json([
                'success' => false,
                'message' => 'Credit card not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $creditCard
        ], 200);
    }

    /**
     * Update an existing credit card.
     *
     * @param int $id
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update($id, Request $request)
    {
        // Validate the request data
        $validator = Validator::make($request->all(), [
            'card_number_encrypted' => 'sometimes|required|string',
            'cardholder_name' => 'sometimes|required|string',
            'expiry_date' => 'sometimes|required|date_format:m/y',
            'last_four_digits' => 'sometimes|required|string|min:4|max:4',
            'is_default' => 'boolean',
            'image' => 'nullable|string',
            'card_type' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Get the authenticated user's ID
        $userId = $request->user()->id;

        // Find the credit card by ID and ensure it belongs to the user
        $creditCard = CreditCard::where('id', $id)->where('user_id', $userId)->first();

        if (!$creditCard) {
            return response()->json([
                'success' => false,
                'message' => 'Credit card not found'
            ], 404);
        }

        // Update the credit card
        $creditCard->update($request->only([
            'card_number_encrypted',
            'cardholder_name',
            'expiry_date',
            'last_four_digits',
            'is_default',
            'image',
            'card_type',
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Credit card updated successfully',
            'data' => $creditCard
        ], 200);
    }

    /**
     * Delete a credit card.
     *
     * @param int $id
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id, Request $request)
    {
        // Get the authenticated user's ID
        $userId = $request->user()->id;

        // Find the credit card by ID and ensure it belongs to the user
        $creditCard = CreditCard::where('id', $id)->where('user_id', $userId)->first();

        if (!$creditCard) {
            return response()->json([
                'success' => false,
                'message' => 'Credit card not found'
            ], 404);
        }

        // Delete the credit card
        $creditCard->delete();

        return response()->json([
            'success' => true,
            'message' => 'Credit card deleted successfully'
        ], 200);
    }
}