<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API;

use App\Http\Resources\CartResource;
use App\Models\Cart;
use App\Models\Product;
use App\Models\CartItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Resources\CartItemResource;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class CartController extends Controller
{
    /**
     * إضافة عنصر إلى السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addToCart(Request $request)
    {
        try {
            $validated = $request->validate([
                'product_id' => 'required|exists:products,id',
                'quantity' => 'required|integer|min:1',
            ]);

            /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
            $user = $request->user();

            if (!$user) {
                return response()->json(['message' => 'غير مصرح'], 401);
            }

            $cart = Cart::firstOrCreate(
                ['user_id' => $user->id],
                ['user_id' => $user->id, 'total_price' => 0]
            );

            $product = Product::findOrFail($validated['product_id']);

            $cartItem = $cart->cart_items()->where('product_id', $validated['product_id'])->first();

            if ($cartItem) {
                $cartItem->update([
                    'quantity' => $cartItem->quantity + $validated['quantity'],
                ]);
            } else {
                $cartItem = $cart->cart_items()->create([
                    'product_id' => $validated['product_id'],
                    'quantity' => $validated['quantity'],
                ]);
            }
            $this->updateCartTotal($cart);
            return response()->json([
                'message' => 'تم إضافة العنصر إلى السلة بنجاح',
                'cart_item' => new CartItemResource($cartItem),
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => 'خطأ في التحقق من البيانات',
                'errors' => $e->errors(),
            ], 422);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'message' => 'المنتج غير موجود',
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error adding item to cart: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'message' => 'حدث خطأ أثناء إضافة العنصر إلى السلة',
            ], 500);
        }
    }

    /**
     * إزالة عنصر من السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeFromCart(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        // البحث عن سلة المستخدم
        $cart = $user->cart()->first();

        if (!$cart) {
            return response()->json(['message' => 'السلة غير موجودة'], 404);
        }

        // البحث عن العنصر في السلة
        $cartItem = $cart->cart_items()->where('product_id', $validated['product_id'])->first();

        if (!$cartItem) {
            return response()->json(['message' => 'العنصر غير موجود في السلة'], 404);
        }

        // حذف العنصر
        $cartItem->delete();

        // تحديث السعر الإجمالي للسلة
        $this->updateCartTotal($cart);

        return response()->json(['message' => 'تم إزالة العنصر من السلة بنجاح']);
    }

    /**
     * جلب جميع عناصر السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCart(Request $request)
    {
        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        $cart = $user->cart()->with('cart_items.product')->first();

        if (!$cart) {
            return response()->json([
                'message' => 'السلة فارغة',
                'cart' => null,
            ]);
        }

        return response()->json([
            'message' => 'تم جلب السلة بنجاح',
            'cart' => new CartResource($cart),
            'count_items' => $cart->cart_items()->count(),

        ]);
    }

    /**
     * تحديث السعر الإجمالي للسلة
     *
     * @param Cart $cart
     * @return void
     */
    protected function updateCartTotal(Cart $cart)
    {
        $total = $cart->cart_items()->with('product')->get()->sum(function ($item) {
            return $item->quantity * ($item->product->price ?? 0);
        });

        $cart->update(['total_price' => $total]);
    }
}
