<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API;

use App\Models\Banner;
use App\Models\Slider;
use App\Models\Partner;
use App\Models\Product;
use Illuminate\Http\Request;
use App\Models\Advertisement;
use App\Http\Controllers\Controller;
use App\Http\Resources\BannerResource;
use App\Http\Resources\SliderResource;
use App\Http\Resources\PartnerResource;
use App\Http\Resources\ProductResource;
use App\Http\Resources\AdvertisementResource;

class ProductController extends Controller
{
    /**
     * Get all products with filtering options
     *
     * @param Request $request
     * @return \Illuminate\Http\Resources\Json\AnonymousResourceCollection
     */
    public function index(Request $request)
    {
        try {
            $productsQuery = Product::with([
                'category',
                'status',
                'cart_items',
                'discounts',
                'orderItems',
                'reviews'
            ])->where('status_id', 1);

            $search = $request->query('search', '');
            $sort = $request->query('sort', 'recent');
            $perPage = $request->query('per_page', 15);
            $page = $request->query('page', 1);

            if (!empty($search)) {
                $productsQuery->where(function ($query) use ($search) {
                    $query->where('name->ar', 'like', '%' . $search . '%')
                        ->orWhere('name->en', 'like', '%' . $search . '%');
                });
            }

            $columns = [
                'products.id',
                'products.category_id',
                'products.sku',
                'products.status_id',
                'products.name',
                'products.description',
                'products.image',
                'products.price',
                'products.created_at',
                'products.updated_at',
            ];

            if ($sort === 'best_selling') {
                $productsQuery->join('order_items', 'products.id', '=', 'order_items.product_id')
                    ->select($columns)
                    ->groupBy($columns)
                    ->orderByRaw('COUNT(order_items.product_id) DESC');
            } else {
                switch ($sort) {
                    case 'high_price':
                        $productsQuery->orderBy('price', 'desc');
                        break;
                    case 'low_price':
                        $productsQuery->orderBy('price', 'asc');
                        break;
                    case 'recent':
                        $productsQuery->orderBy('created_at', 'desc');
                        break;
                    default:
                        $productsQuery->orderBy('created_at', 'desc');
                        break;
                }
                $productsQuery->select($columns);
            }

            $products = $productsQuery->paginate($perPage, ['*'], 'page', $page);

            $sliders = Slider::where('is_active', 1)->get();
            $partners = Partner::all();
            $advertisements = Advertisement::where('location_type', 'product')->where('locatable_type', null)->where('key', 1)->get();

            return response()->json([
                'message' => 'تم جلب البيانات بنجاح',
                'products' => ProductResource::collection($products->items()),
                'advertisements' => AdvertisementResource::collection($advertisements),

                'pagination' => [
                    'current_page' => $products->currentPage(),
                    'per_page' => $products->perPage(),
                    'total' => $products->total(),
                    'last_page' => $products->lastPage(),
                    'next_page_url' => $products->nextPageUrl(),
                    'prev_page_url' => $products->previousPageUrl(),
                ],
                'sliders' => SliderResource::collection($sliders),
                'partners' => PartnerResource::collection($partners),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'حدث خطأ',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    /**
     * Get single product details with all relationships
     *
     * @param Product $product
     * @return ProductResource
     */
    public function show($id)
    {
        $product = Product::with([
            'category',
            'status',
            'cartItems',
            'discounts',
            'orderItems',
            'reviews',
            'favourites'
        ])->find($id);

        if (!$product) {
            return response()->json([
                'message' => 'Product not found'
            ], 404);
        }

        return response()->json([
            'products' => new ProductResource($product),
            'advertisements' => AdvertisementResource::collection($product->advertisements),
        ], 200);
    }
}