<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use Carbon\Carbon;
use App\Models\Tag;
use App\Models\Order;
use App\Models\Status;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\TagResource;
use App\Http\Controllers\Controller;
use App\Http\Resources\Dashboard\OrderResource;
use App\Http\Resources\ProductResource;
use Illuminate\Support\Facades\Storage;
use App\Http\Resources\CategoryResource;
use Illuminate\Support\Facades\Validator;

class ServiceController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::where('type', 1)->with(['category', 'status', 'cartItems', 'discounts', 'orderItems', 'reviews', 'images', 'tags']);

        if ($period = $request->input('period')) {
            $validator = Validator::make(['period' => $period], [
                'period' => 'in:24_hours,7_days,30_days,12_months'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'قيمة الفترة غير صحيحة، استخدم 24_hours, 7_days, 30_days, أو 12_months'
                ], 400);
            }

            $startDate = match ($period) {
                '24_hours' => Carbon::now()->subHours(24),
                '7_days' => Carbon::now()->subDays(7),
                '30_days' => Carbon::now()->subDays(30),
                '12_months' => Carbon::now()->subMonths(12),
            };

            $query->where('created_at', '>=', $startDate);
        } elseif ($date = $request->input('date')) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy'
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $query->whereDate('created_at', $formattedDate);
        }

        if ($statusId = $request->input('status_id')) {
            $validator = Validator::make(['status_id' => $statusId], [
                'status_id' => 'integer|exists:statuses,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم حالة المنتج غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('status_id', $statusId);
        }

        if ($productId = $request->input('product_id')) {
            $validator = Validator::make(['product_id' => $productId], [
                'product_id' => 'integer|exists:products,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم المنتج غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('id', $productId);
        }

        $products = $query->paginate(10);

        return response()->json([
            'message' => 'تم استرجاع المنتجات بنجاح',
            'products' => ProductResource::collection($products),
            'total' => $products->total(),
            'current_page' => $products->currentPage(),
            'last_page' => $products->lastPage()
        ], 200);
    }

    public function show($id)
    {
        try {
            $product = Product::find($id);
            if (!$product) {
                return response()->json([
                    'status' => false,
                    'message' => 'المنتج غير موجود',
                    'error' => 'Product not found',
                ], 404, [], JSON_UNESCAPED_UNICODE);
            }

            $orders = Order::whereHas('order_items', function ($query) use ($id) {
                $query->where('product_id', $id);
            })
                ->with(['order_items.product', 'status', 'user'])
                ->whereIn('status_id', [8, 9, 10]) // Completed orders: accepted, delivered, on-process
                ->get();

            return response()->json([
                'status' => true,
                'message' => 'تم استرجاع جميع الطلبات للمنتج بنجاح',
                'data' => [
                    'product' => new ProductResource($product),
                    'orders' => OrderResource::collection($orders ?: collect([])), // Ensure empty collection
                ],
            ], 200, [], JSON_UNESCAPED_UNICODE);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => false,
                'message' => 'المنتج غير موجود',
                'error' => 'Product not found',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'حدث خطأ أثناء استرجاع الطلبات',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    public function getCreateEditData()
    {
        try {
            $categories = Category::select('id', 'name')->get();
            $statuses = Status::select('id', 'name')->whereBetween('id', [1, 4])->get();
            $tags = Tag::select('id', 'name')->get();

            return response()->json([
                'message' => 'تم استرجاع بيانات المنتج بنجاح',
                'categories' => CategoryResource::collection($categories),
                'statuses' => $statuses,
                'tags' => TagResource::collection($tags)
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'حدث خطأ: ' . $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'category_id' => 'required|integer|exists:categories,id',
                'sku' => 'required|string|max:255|unique:products,sku',
                'status_id' => 'required|integer|exists:statuses,id',
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'short_description' => 'nullable|string|max:500',
                'image' => 'required|file|image|mimes:jpg,jpeg,png|max:2048',
                'price' => 'required|numeric|min:0',
                'stock' => 'required|integer|min:0',
                'publish_at' => 'nullable|date',
                'visibility' => 'required|in:public,private,draft',
                'manufacturer_name' => 'nullable|string|max:255',
                'manufacturer_brand' => 'nullable|string|max:255',
                'label' => 'nullable|string|max:255',
                'gallery' => 'nullable|array',
                'gallery.*' => 'file|image|mimes:jpg,jpeg,png|max:2048',
                'tags' => 'nullable|array',
                'tags.*' => 'integer|exists:tags,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'خطأ في التحقق من البيانات',
                    'errors' => $validator->errors()
                ], 422);
            }

            DB::beginTransaction();

            $product = Product::create($request->only([
                'category_id',
                'sku',
                'status_id',
                'name',
                'description',
                'short_description',
                'price',
                'stock',
                'publish_at',
                'visibility',
                'manufacturer_name',
                'manufacturer_brand',
                'label'
            ]) + ['type' => 1]);

            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('products', 'public');
                $product->update([
                    'image' => $imagePath,
                ]);
            }

            // Store gallery images
            if ($request->hasFile('gallery')) {
                foreach ($request->file('gallery') as $galleryImage) {
                    $galleryPath = $galleryImage->store('products', 'public');
                    $product->images()->create([
                        'url' => $galleryPath,
                    ]);
                }
            }

            if ($request->has('tags')) {
                $product->tags()->sync($request->input('tags'));
            }

            DB::commit();

            return response()->json([
                'message' => 'تم إنشاء المنتج بنجاح',
                'product' => new ProductResource(
                    Product::with(['category', 'status', 'images', 'tags'])->findOrFail($product->id)
                )
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'حدث خطأ: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            // Validate the request with nullable fields
            $validator = Validator::make($request->all(), [
                'category_id' => 'nullable|integer|exists:categories,id',
                'sku' => 'nullable|string|max:255|unique:products,sku,' . $id,
                'status_id' => 'nullable|integer|exists:statuses,id',
                'name' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'short_description' => 'nullable|string|max:500',
                'image' => 'nullable|file|image|mimes:jpg,jpeg,png|max:2048',
                'price' => 'nullable|numeric|min:0',
                'stock' => 'nullable|integer|min:0',
                'publish_at' => 'nullable|date',
                'visibility' => 'nullable|in:public,private,draft',
                'manufacturer_name' => 'nullable|string|max:255',
                'manufacturer_brand' => 'nullable|string|max:255',
                'label' => 'nullable|string|max:255',
                'gallery' => 'nullable|array',
                'gallery.*' => 'nullable|file|image|mimes:jpg,jpeg,png|max:2048',
                'tags' => 'nullable|array',
                'tags.*' => 'nullable|integer|exists:tags,id',
            ], [
                'category_id.integer' => 'معرف الفئة يجب أن يكون رقمًا.',
                'category_id.exists' => 'الفئة المحددة غير موجودة.',
                'sku.string' => 'كود SKU يجب أن يكون نصًا.',
                'sku.max' => 'كود SKU يجب ألا يتجاوز 255 حرفًا.',
                'sku.unique' => 'كود SKU مستخدم بالفعل.',
                'status_id.integer' => 'معرف الحالة يجب أن يكون رقمًا.',
                'status_id.exists' => 'الحالة المحددة غير موجودة.',
                'name.string' => 'الاسم يجب أن يكون نصًا.',
                'name.max' => 'الاسم يجب ألا يتجاوز 255 حرفًا.',
                'description.string' => 'الوصف يجب أن يكون نصًا.',
                'short_description.string' => 'الوصف المختصر يجب أن يكون نصًا.',
                'short_description.max' => 'الوصف المختصر يجب ألا يتجاوز 500 حرف.',
                'image.file' => 'يجب أن تكون الصورة ملفًا.',
                'image.image' => 'يجب أن تكون الصورة صورة صالحة.',
                'image.mimes' => 'الصورة يجب أن تكون بصيغة jpg، jpeg، أو png.',
                'image.max' => 'حجم الصورة يجب ألا يتجاوز 2 ميجابايت.',
                'price.numeric' => 'السعر يجب أن يكون رقمًا.',
                'price.min' => 'السعر يجب ألا يكون أقل من 0.',
                'stock.integer' => 'المخزون يجب أن يكون رقمًا صحيحًا.',
                'stock.min' => 'المخزون يجب ألا يكون أقل من 0.',
                'publish_at.date' => 'تاريخ النشر يجب أن يكون تاريخًا صالحًا.',
                'visibility.in' => 'الرؤية يجب أن تكون إما عامة، خاصة، أو مسودة.',
                'manufacturer_name.string' => 'اسم الصانع يجب أن يكون نصًا.',
                'manufacturer_name.max' => 'اسم الصانع يجب ألا يتجاوز 255 حرفًا.',
                'manufacturer_brand.string' => 'العلامة التجارية يجب أن تكون نصًا.',
                'manufacturer_brand.max' => 'العلامة التجارية يجب ألا يتجاوز 255 حرفًا.',
                'label.string' => 'الملصق يجب أن يكون نصًا.',
                'label.max' => 'الملصق يجب ألا يتجاوز 255 حرفًا.',
                'gallery.array' => 'معرض الصور يجب أن يكون مصفوفة.',
                'gallery.*.file' => 'كل صورة في المعرض يجب أن تكون ملفًا.',
                'gallery.*.image' => 'كل صورة في المعرض يجب أن تكون صورة صالحة.',
                'gallery.*.mimes' => 'صور المعرض يجب أن تكون بصيغة jpg، jpeg، أو png.',
                'gallery.*.max' => 'حجم كل صورة في المعرض يجب ألا يتجاوز 2 ميجابايت.',
                'tags.array' => 'العلامات يجب أن تكون مصفوفة.',
                'tags.*.integer' => 'كل علامة يجب أن تكون رقمًا.',
                'tags.*.exists' => 'إحدى العلامات المحددة غير موجودة.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'خطأ في التحقق من البيانات',
                    'errors' => $validator->errors(),
                ], 422);
            }

            $product = Product::findOrFail($id);

            DB::beginTransaction();

            // Prepare data for update, only including provided fields
            $updateData = collect($request->only([
                'category_id',
                'sku',
                'status_id',
                'name',
                'description',
                'short_description',
                'price',
                'stock',
                'publish_at',
                'visibility',
                'manufacturer_name',
                'manufacturer_brand',
                'label',
            ]))->filter()->toArray();

            // Update only provided fields
            if (!empty($updateData)) {
                $product->update($updateData);
            }

            // Handle main image update if provided
            if ($request->hasFile('image') && $request->file('image')->isValid()) {
                // Delete existing main image
                $product->images()->delete();
                $imagePath = $request->file('image')->store('products', 'public');
                $product->update(['image' => $imagePath]);
            }

            // Handle gallery images if provided
            if ($request->hasFile('gallery')) {
                // Delete existing gallery images
                $product->images()->delete();
                foreach ($request->file('gallery') as $galleryImage) {
                    if ($galleryImage->isValid()) {
                        $galleryPath = $galleryImage->store('products', 'public');
                        $product->images()->create([
                            'url' => $galleryPath,
                        ]);
                    }
                }
            }

            // Sync tags if provided
            if ($request->has('tags')) {
                $product->tags()->sync($request->input('tags', []));
            }

            DB::commit();

            return response()->json([
                'message' => 'تم تعديل المنتج بنجاح',
                'product' => new ProductResource(
                    Product::with(['category', 'status', 'images', 'tags'])->findOrFail($product->id)
                ),
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'حدث خطأ: ' . $e->getMessage(),
            ], 500);
        }
    }
    public function destroy($id)
    {
        try {
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|exists:products,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم المنتج غير صحيح أو غير موجود'
                ], 400);
            }

            $product = Product::findOrFail($id);

            DB::beginTransaction();

            foreach ($product->images as $image) {
                Storage::disk('public')->delete($image->url);
                $image->delete();
            }

            $product->tags()->detach();

            $product->delete();

            DB::commit();

            return response()->json([
                'message' => 'تم حذف المنتج بنجاح'
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'حدث خطأ: ' . $e->getMessage()
            ], 500);
        }
    }
}