<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use App\Models\Page;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;


class PageController extends Controller
{
    /**
     * Display a listing of all pages with their tags.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        $pages = Page::with('tags')->get();

        return response()->json([
            'message' => 'تم استرجاع الصفحات بنجاح',
            'data' => $pages,
        ], 200);
    }

    /**
     * Store a newly created page in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'navigation_bar' => 'required|string|max:255',
            'page_name' => 'required|string|max:255',
            'page_category' => 'required|in:home,about,services,contact,blog',
            'status' => 'required|in:draft,published,archived',
            'visibility' => 'required|in:public,private,restricted',
            'publish_schedule' => 'nullable|date',
            'description' => 'nullable|string',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'بيانات الإدخال غير صحيحة',
                'errors' => $validator->errors(),
            ], 400);
        }

        $page = Page::create($request->only([
            'navigation_bar',
            'page_name',
            'page_category',
            'status',
            'visibility',
            'publish_schedule',
            'description',
        ]));

        if ($request->has('tags')) {
            $page->tags()->createMany(
                array_map(fn($tag) => ['value' => $tag], $request->tags)
            );
        }

        return response()->json([
            'message' => 'تم إنشاء الصفحة بنجاح',
            'data' => $page->load('tags'),
        ], 201);
    }

    /**
     * Display the specified page with its tags.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $page = Page::with('tags')->find($id);

        if (!$page) {
            return response()->json([
                'message' => 'الصفحة غير موجودة',
            ], 404);
        }

        return response()->json([
            'message' => 'تم استرجاع الصفحة بنجاح',
            'data' => $page,
        ], 200);
    }

    /**
     * Update the specified page in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $page = Page::find($id);

        if (!$page) {
            return response()->json([
                'message' => 'الصفحة غير موجودة',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'navigation_bar' => 'required|string|max:255',
            'page_name' => 'required|string|max:255',
            'page_category' => 'required|in:home,about,services,contact,blog',
            'status' => 'required|in:draft,published,archived',
            'visibility' => 'required|in:public,private,restricted',
            'publish_schedule' => 'nullable|date',
            'description' => 'nullable|string',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'بيانات الإدخال غير صحيحة',
                'errors' => $validator->errors(),
            ], 400);
        }

        $page->update($request->only([
            'navigation_bar',
            'page_name',
            'page_category',
            'status',
            'visibility',
            'publish_schedule',
            'description',
        ]));

        if ($request->has('tags')) {
            $page->tags()->delete();
            $page->tags()->createMany(
                array_map(fn($tag) => ['value' => $tag], $request->tags)
            );
        }

        return response()->json([
            'message' => 'تم تحديث الصفحة بنجاح',
            'data' => $page->load('tags'),
        ], 200);
    }

    /**
     * Remove the specified page from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $page = Page::find($id);

        if (!$page) {
            return response()->json([
                'message' => 'الصفحة غير موجودة',
            ], 404);
        }

        $page->delete(); // Tags are automatically deleted via ON DELETE CASCADE

        return response()->json([
            'message' => 'تم حذف الصفحة بنجاح',
        ], 200);
    }
}