<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use App\Models\Cart;
use App\Models\User;
use App\Models\Order;
use App\Models\Status;
use App\Models\Product;
use App\Models\Discount;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Resources\OrderResource;
use App\Http\Resources\StatusResource;
use Illuminate\Support\Facades\Validator;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Order::with(['user', 'status', 'order_items']);

        if ($period = $request->input('period')) {
            $validator = Validator::make(['period' => $period], [
                'period' => 'in:24_hours,7_days,30_days,12_months'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'قيمة الفترة غير صحيحة، استخدم 24_hours, 7_days, 30_days, أو 12_months'
                ], 400);
            }

            $startDate = match ($period) {
                '24_hours' => Carbon::now()->subHours(24),
                '7_days' => Carbon::now()->subDays(7),
                '30_days' => Carbon::now()->subDays(30),
                '12_months' => Carbon::now()->subMonths(12),
            };

            $query->where('created_at', '>=', $startDate);
        } elseif ($date = $request->input('date')) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy'
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $query->whereDate('created_at', $formattedDate);
        }

        if ($statusId = $request->input('status_id')) {
            $validator = Validator::make(['status_id' => $statusId], [
                'status_id' => 'integer|exists:statuses,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم حالة الطلب غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('status_id', $statusId);
        }

        if ($orderId = $request->input('order_id')) {
            $validator = Validator::make(['order_id' => $orderId], [
                'order_id' => 'integer|exists:orders,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم الطلب غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('id', $orderId);
        }

        $orders = $query->paginate(10);

        return response()->json([
            'message' => 'تم استرجاع الطلبات بنجاح',
            'orders' => OrderResource::collection($orders),
            'total' => $orders->total(),
            'current_page' => $orders->currentPage(),
            'last_page' => $orders->lastPage()
        ], 200);
    }

    public function changeStatus(Request $request, $id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'message' => 'الطلب غير موجود'
            ], 404);
        }

        $request->validate([
            'status_id' => 'required|integer|exists:statuses,id', // تأكد إن الحالة موجودة في جدول statuses
        ]);

        // لو الحالة القديمة زي الحالة الجديدة
        if ($order->status_id == $request->status_id) {
            return response()->json([
                'message' => 'الحالة لم تتغير لأن الطلب بالفعل في هذه الحالة'
            ], 400);
        }

        $order->status_id = $request->status_id;
        $order->save();

        return response()->json([
            'message' => 'تم تغيير حالة الطلب بنجاح',
            'order' => new OrderResource($order)
        ], 200);
    }

    public function newOrders(Request $request)
    {
        $query = Order::with(['user', 'status', 'order_items'])
            ->where('status_id', 1);

        if ($period = $request->input('period')) {
            $validator = Validator::make(['period' => $period], [
                'period' => 'in:24_hours,7_days,30_days,12_months'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'قيمة الفترة غير صحيحة، استخدم 24_hours, 7_days, 30_days, أو 12_months'
                ], 400);
            }

            $startDate = match ($period) {
                '24_hours' => Carbon::now()->subHours(24),
                '7_days' => Carbon::now()->subDays(7),
                '30_days' => Carbon::now()->subDays(30),
                '12_months' => Carbon::now()->subMonths(12),
            };

            $query->where('created_at', '>=', $startDate);
        } elseif ($date = $request->input('date')) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy'
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $query->whereDate('created_at', $formattedDate);
        }

        if ($orderId = $request->input('order_id')) {
            $validator = Validator::make(['order_id' => $orderId], [
                'order_id' => 'integer|exists:orders,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم الطلب غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('id', $orderId);
        }

        $orders = $query->paginate(10);

        return response()->json([
            'message' => 'تم استرجاع الطلبات الجديدة بنجاح',
            'orders' => OrderResource::collection($orders),
            'total' => $orders->total(),
            'current_page' => $orders->currentPage(),
            'last_page' => $orders->lastPage()
        ], 200);
    }

    public function canceledOrders(Request $request)
    {
        $query = Order::with(['user', 'status', 'order_items']);

        if ($statusId = $request->input('status_id')) {
            $validator = Validator::make(['status_id' => $statusId], [
                'status_id' => 'integer|exists:statuses,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم حالة الطلب غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('status_id', $statusId);
        } else {
            $query->where('status_id', 4);
        }

        if ($period = $request->input('period')) {
            $validator = Validator::make(['period' => $period], [
                'period' => 'in:24_hours,7_days,30_days,12_months'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'قيمة الفترة غير صحيحة، استخدم 24_hours, 7_days, 30_days, أو 12_months'
                ], 400);
            }

            $startDate = match ($period) {
                '24_hours' => Carbon::now()->subHours(24),
                '7_days' => Carbon::now()->subDays(7),
                '30_days' => Carbon::now()->subDays(30),
                '12_months' => Carbon::now()->subMonths(12),
            };

            $query->where('created_at', '>=', $startDate);
        } elseif ($date = $request->input('date')) {
            $validator = Validator::make(['date' => $date], [
                'date' => 'date_format:m/d/Y'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'صيغة التاريخ غير صحيحة، استخدم mm/dd/yyyy'
                ], 400);
            }

            $formattedDate = Carbon::createFromFormat('m/d/Y', $date)->format('Y-m-d');
            $query->whereDate('created_at', $formattedDate);
        }

        if ($orderId = $request->input('order_id')) {
            $validator = Validator::make(['order_id' => $orderId], [
                'order_id' => 'integer|exists:orders,id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => 'رقم الطلب غير صحيح أو غير موجود'
                ], 400);
            }

            $query->where('id', $orderId);
        }

        $orders = $query->paginate(10);

        return response()->json([
            'message' => 'تم استرجاع الطلبات الملغاة بنجاح',
            'orders' => OrderResource::collection($orders),
            'total' => $orders->total(),
            'current_page' => $orders->currentPage(),
            'last_page' => $orders->lastPage()
        ], 200);
    }

    public function getStatus()
    {
        return response()->json([
            'message' => 'تم استرجاع الحالات بنجاح',
            'status' => StatusResource::collection(Status::whereBetween('id', [1, 4])->get())
        ], 200);
    }


}