<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use App\Models\Faq;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\FaqResource;
use Illuminate\Support\Facades\Validator;

class FaqController extends Controller
{
    /**
     * Display a listing of FAQs and Help entries.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $faqs = Faq::where('type', $request->type)->get();

        return response()->json([
            'message' => 'تم استرجاع الأسئلة الشائعة وسجلات المساعدة بنجاح',
            'data' => FaqResource::collection($faqs),
        ], 200);
    }

    /**
     * Store a newly created FAQ or Help entry in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'type' => 'required|in:1,2',
            'question_en' => 'required|string|max:255',
            'question_ar' => 'nullable|string|max:255',
            'answer_en' => 'required|string',
            'answer_ar' => 'nullable|string',
            'active' => 'boolean',
            'sort' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'بيانات الإدخال غير صحيحة',
                'errors' => $validator->errors(),
            ], 400);
        }

        $faq = Faq::create([
            'type' => $request->type,

            'question' => json_encode([
                'en' => $request->question_en,
                'ar' => $request->question_ar
            ], JSON_UNESCAPED_UNICODE),

            'answer' => json_encode([
                'en' => $request->answer_en,
                'ar' => $request->answer_ar
            ], JSON_UNESCAPED_UNICODE),

            'active' => $request->input('active', true),
            'sort' => $request->sort,
        ]);


        return response()->json([
            'message' => 'تم إنشاء السجل بنجاح',
            'data' => $faq,
        ], 201);
    }

    /**
     * Display the specified FAQ or Help entry.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        $faq = Faq::find($id);

        if (!$faq) {
            return response()->json([
                'message' => 'السجل غير موجود',
            ], 404);
        }

        return response()->json([
            'message' => 'تم استرجاع السجل بنجاح',
            'data' => $faq,
        ], 200);
    }

    /**
     * Update the specified FAQ or Help entry in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $faq = Faq::find($id);

        if (!$faq) {
            return response()->json([
                'message' => 'السجل غير موجود',
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'question_en' => 'required|string|max:255',
            'question_ar' => 'nullable|string|max:255',
            'answer_en' => 'required|string',
            'answer_ar' => 'nullable|string',
            'active' => 'boolean',
            'sort' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'بيانات الإدخال غير صحيحة',
                'errors' => $validator->errors(),
            ], 400);
        }

        $faq->update([
            'question' => json_encode([
                'en' => $request->question_en,
                'ar' => $request->question_ar
            ], JSON_UNESCAPED_UNICODE),

            'answer' => json_encode([
                'en' => $request->answer_en,
                'ar' => $request->answer_ar
            ], JSON_UNESCAPED_UNICODE),

            'active' => $request->input('active', $faq->active),
            'sort' => $request->sort,
        ]);


        return response()->json([
            'message' => 'تم تحديث السجل بنجاح',
            'data' => new FaqResource($faq),
        ], 200);
    }

    /**
     * Remove the specified FAQ or Help entry from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        $faq = Faq::find($id);

        if (!$faq) {
            return response()->json([
                'message' => 'السجل غير موجود',
            ], 404);
        }

        $faq->delete();

        return response()->json([
            'message' => 'تم حذف السجل بنجاح',
        ], 200);
    }
}