<?php

/*
    Created by Mohamed.
*/

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
class AuthController extends Controller
{
    /**
     * Handle admin login
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        $admin = Admin::where('email', $request->email)->first();

        if (!$admin || !Hash::check($request->password, $admin->password)) {
            return response()->json([
                'message' => 'Invalid login credentials',
            ], 401);
        }

        // if (!$admin->hasVerifiedEmail()) {
        //     return response()->json([
        //         'message' => 'Please verify your email first',
        //     ], 403);
        // }

        $token = $admin->createToken('admin-token')->plainTextToken;

        return response()->json([
            'message' => 'Login successful',
            'admin' => $admin,
            'token' => $token,
        ], 200);
    }

    /**
     * Send email verification link or OTP
     */
    public function sendResetOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:admins,email',
        ], [
            'email.required' => 'البريد الإلكتروني مطلوب.',
            'email.email' => 'صيغة البريد الإلكتروني غير صحيحة.',
            'email.exists' => 'البريد الإلكتروني غير مسجل.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'فشل التحقق من البيانات.',
                'errors' => $validator->errors(),
            ], 422);
        }

        $admin = Admin::where('email', $request->email)->first();
        $cooldownKey = 'reset_otp_sent_' . $admin->email;

        if (Cache::has($cooldownKey)) {
            return response()->json([
                'message' => 'تم إرسال رمز التحقق مؤخرًا، الرجاء الانتظار دقيقة.',
            ], 429);
        }

        $otp = rand(100000, 999999);
        Cache::put('reset_otp_' . $admin->email, $otp, now()->addMinutes(10));
        Cache::put($cooldownKey, true, now()->addMinute());

        try {
            $admin->notify(new \App\Notifications\ResetPasswordOtpNotification($otp));
            return response()->json(['message' => 'تم إرسال رمز التحقق إلى بريدك.'], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'فشل في إرسال رمز التحقق.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    public function verifyResetOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:admins,email',
            'otp' => 'required|numeric|digits:6',
        ], [
            'email.required' => 'البريد الإلكتروني مطلوب.',
            'otp.required' => 'رمز التحقق مطلوب.',
            'otp.numeric' => 'رمز التحقق يجب أن يكون رقمًا.',
            'otp.digits' => 'رمز التحقق يجب أن يتكون من 6 أرقام.',
            'email.exists' => 'البريد الإلكتروني غير مسجل.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'فشل التحقق من البيانات.',
                'errors' => $validator->errors(),
            ], 422);
        }

        $admin = Admin::where('email', $request->email)->first();
        $cachedOtp = Cache::get('reset_otp_' . $admin->email);

        if (!$cachedOtp || $cachedOtp != $request->otp) {
            return response()->json([
                'message' => 'رمز التحقق غير صحيح أو منتهي.',
            ], 400);
        }

        Cache::put('otp_verified_' . $admin->email, true, now()->addMinutes(10));

        return response()->json([
            'message' => 'تم التحقق من الرمز. يمكنك الآن تعيين كلمة مرور جديدة.',
        ], 200);
    }





    /**
     * Verify OTP for email verification
     */


    /**
     * Handle password reset request and update
     */

    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:admins,email',
            'password' => 'required|string|min:8|confirmed',
        ], [
            'email.required' => 'البريد الإلكتروني مطلوب.',
            'password.required' => 'كلمة المرور مطلوبة.',
            'password.min' => 'كلمة المرور يجب أن تكون 8 أحرف على الأقل.',
            'password.confirmed' => 'تأكيد كلمة المرور غير مطابق.',
            'email.exists' => 'البريد الإلكتروني غير مسجل.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'فشل التحقق من البيانات.',
                'errors' => $validator->errors(),
            ], 422);
        }

        $admin = Admin::where('email', $request->email)->first();

        if (!Cache::get('otp_verified_' . $admin->email)) {
            return response()->json([
                'message' => 'لم يتم التحقق من البريد الإلكتروني أو انتهت صلاحية الرمز.',
            ], 403);
        }

        $admin->update([
            'password' => Hash::make($request->password),
            'remember_token' => Str::random(60),
        ]);

        Cache::forget('otp_verified_' . $admin->email);
        Cache::forget('reset_otp_' . $admin->email);
        Cache::forget('reset_otp_sent_' . $admin->email);

        return response()->json([
            'message' => 'تم تغيير كلمة المرور بنجاح.',
        ], 200);
    }


    /**
     * Change password for authenticated admin
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        $admin = Auth::guard('admin')->user();

        if (!$admin || !Hash::check($request->current_password, $admin->password)) {
            return response()->json([
                'message' => 'Current password is incorrect',
            ], 401);
        }

        $admin->forceFill([
            'password' => Hash::make($request->new_password),
            //'remember_token' => Str::random(60),
        ])->save();

        return response()->json([
            'message' => 'Password changed successfully',
        ], 200);
    }
    public function getUser()
    {
        return response()->json([
            'message' => 'get data success',
            'data' => auth()->user()
        ], 200);
    }
}