<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API\Dashboard;

use Illuminate\Http\Request;
use App\Models\Advertisement;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class AdvertisementController extends Controller
{
    /**
     * Retrieve primary and secondary advertisements.
     *
     * @return JsonResponse
     */
    public function index(Request $request)
    {
        if ($request->type == 'home') {
            $ads = Advertisement::where('type', 'home')->first();
        } else {
            $ads = Advertisement::where('type', 'product')->get();
        }

        return response()->json([
            'message' => 'Advertisements retrieved successfully',
            'data' => [
                'ads' => $ads,

            ],
        ], 200);
    }
    /**
     * Create a new advertisement.
     *
     * @param Request $request
     * @return JsonResponse
     */

    public function store(Request $request): JsonResponse
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'title' => 'required|array', // Title must be an array (JSON object)
                'title.en' => 'required|string|max:255', // English title required
                'title.ar' => 'required|string|max:255', // Arabic title required
                'description' => 'nullable|array', // Description is optional array
                'description.en' => 'nullable|string', // English description optional
                'description.ar' => 'nullable|string', // Arabic description optional
                'image' => 'required|image|mimes:jpg,png,jpeg|max:2048', // Image file, max 2MB
                'key' => 'required|boolean',
                'product_id' => 'required|integer|exists:products,id',
                'color' => [
                    'required',
                    'string',
                    'max:255',
                    'regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/'
                ],
                'type' => 'required|in:home,product',
                'locatable_id' => 'nullable|integer|min:0',
            ], [
                'title.required' => 'عنوان الإعلان مطلوب',
                'title.array' => 'عنوان الإعلان يجب أن يكون كائن JSON',
                'title.en.required' => 'عنوان الإعلان بالإنجليزية مطلوب',
                'title.en.string' => 'عنوان الإعلان بالإنجليزية يجب أن يكون نصًا',
                'title.ar.required' => 'عنوان الإعلان بالعربية مطلوب',
                'title.ar.string' => 'عنوان الإعلان بالعربية يجب أن يكون نصًا',
                'description.array' => 'وصف الإعلان يجب أن يكون كائن JSON',
                'image.required' => 'صورة الإعلان مطلوبة',
                'image.image' => 'يجب أن تكون الصورة من نوع صورة صالحة',
                'image.mimes' => 'يجب أن تكون الصورة بصيغة: jpg أو png أو jpeg',
                'image.max' => 'أقصى حجم مسموح للصورة هو 2 ميجابايت',
                'key.required' => 'حقل التمييز مطلوب',
                'key.boolean' => 'قيمة التمييز يجب أن تكون منطقية',
                'product_id.required' => 'رقم المنتج مطلوب',
                'product_id.integer' => 'رقم المنتج يجب أن يكون رقمًا صحيحًا',
                'product_id.exists' => 'المنتج غير موجود',
                'color.required' => 'لون الخلفية مطلوب',
                'color.regex' => 'صيغة اللون غير صحيحة، يجب أن تكون مثل: #ffffff',
                'type.required' => 'نوع الإعلان مطلوب',
                'type.in' => 'نوع الإعلان يجب أن يكون home أو product',
                'locatable_id.integer' => 'معرّف العنصر يجب أن يكون رقمًا صحيحًا',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'message' => 'فشل التحقق من البيانات',
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Check if primary advertisement already exists when key = 1
            if ($request->key && Advertisement::where('key', true)->exists()) {
                return response()->json([
                    'status' => false,
                    'message' => 'يوجد إعلان أساسي بالفعل',
                ], 409);
            }

            // Handle image file upload
            $imagePath = $request->file('image')->store('advertisements', 'public');

            // Determine locatable_type and location_type based on type
            $locatableType = $request->type === 'product' ? 'App\\Models\\Product' : null;
            $locationType = $request->type;

            // Convert title and description to JSON strings
            $title = json_encode($request->title);
            $description = $request->description ? json_encode($request->description) : null;

            // Create the advertisement
            $advertisement = Advertisement::create([
                'title' => $title,
                'description' => $description,
                'image' => $imagePath,
                'location_type' => $locationType,
                'locatable_id' => $request->locatable_id,
                'locatable_type' => $locatableType,
                'key' => $request->key,
                'product_id' => $request->product_id,
                'color' => $request->color ?? '#ffca1a',
            ]);

            // Generate the full URL for the image
            $imageUrl = Storage::url($imagePath);

            // Decode title and description for response
            $advertisement->title = json_decode($advertisement->title, true);
            $advertisement->description = $advertisement->description ? json_decode($advertisement->description, true) : null;

            return response()->json([
                'status' => true,
                'message' => 'تم إنشاء الإعلان بنجاح',
                'data' => array_merge($advertisement->toArray(), ['image_url' => $imageUrl]),
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'حدث خطأ غير متوقع أثناء إنشاء الإعلان',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    /**
     * Summary of destroy
     * @param mixed $id
     * @return JsonResponse
     */
    public function destroy($id): JsonResponse
    {
        try {
            // Find the advertisement
            $advertisement = Advertisement::find($id);

            if (!$advertisement) {
                return response()->json([
                    'status' => false,
                    'message' => 'الإعلان غير موجود',
                ], 404);
            }

            if ($advertisement->image && Storage::disk('public')->exists($advertisement->image)) {
                Storage::disk('public')->delete($advertisement->image);
            }

            $advertisement->delete();

            return response()->json([
                'status' => true,
                'message' => 'تم حذف الإعلان بنجاح',
                'data' => null,
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'حدث خطأ غير متوقع أثناء حذف الإعلان',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}