<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Address;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    /**
     * إنشاء طلب جديد من السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createOrder(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validated = $request->validate([
            'shipping_address_id' => 'required|exists:addresses,id',
            'payment_method' => 'required|in:credit_card,cash_on_delivery',
        ]);

        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        // التحقق من وجود العنوان وملكيته للمستخدم
        $address = $user->addresses()->where('id', $validated['shipping_address_id'])->first();
        if (!$address) {
            return response()->json(['message' => 'العنوان غير موجود أو لا ينتمي لك'], 404);
        }

        // البحث عن سلة المستخدم
        $cart = $user->cart()->with('cart_items.product')->first();
        if (!$cart || $cart->cart_items->isEmpty()) {
            return response()->json(['message' => 'السلة فارغة'], 400);
        }

        // حساب السعر الإجمالي من عناصر السلة
        $totalPrice = $cart->cart_items->sum(function ($item) {
            return $item->quantity * ($item->product->price ?? 0);
        });

        DB::beginTransaction();

        try {
            // إنشاء الطلب
            $order = Order::create([
                'user_id' => $user->id,
                'full_name' => $user->full_name,
                'phone_number' => $cart->phone_number ?? $user->phone_number ?? null,
                'total_price' => $totalPrice,
                'shipping_address' => json_encode([
                    'address_line_1' => $address->address_line_1,
                    'address_line_2' => $address->address_line_2,
                    'city' => $address->city,
                    'country' => $address->country,
                    'postal_code' => $address->postal_code,
                ]),
                'payment_method' => $validated['payment_method'],
                'status_id' => 1, // افتراضي: قيد الانتظار
            ]);

            // تحويل عناصر السلة إلى عناصر الطلب
            foreach ($cart->cart_items as $cartItem) {
                $order->order_items()->create([
                    'product_id' => $cartItem->product_id,
                    'price' => $cartItem->product->price * $cartItem->quantity,
                ]);
            }

            // إفراغ السلة
            $cart->cart_items()->delete();
            $cart->delete();

            DB::commit();

            return response()->json([
                'message' => 'تم إنشاء الطلب بنجاح',
                'order' => $order->load('order_items.product'),
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'فشل إنشاء الطلب',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}