<?php

/*
    Created by Mohamed.
*/


namespace App\Http\Controllers\API;
use App\Models\Cart;
use App\Models\Product;
use App\Models\CartItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class CartController extends Controller
{
    /**
     * إضافة عنصر إلى السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function addToCart(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
        ]);

        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        // البحث عن سلة المستخدم أو إنشاء واحدة جديدة
        $cart = Cart::firstOrCreate(
            ['user_id' => $user->id],
            ['user_id' => $user->id, 'total_price' => 0]
        );

        // التحقق من وجود المنتج
        $product = Product::findOrFail($validated['product_id']);

        // البحث عن العنصر في السلة
        $cartItem = $cart->cart_items()->where('product_id', $validated['product_id'])->first();

        if ($cartItem) {
            // تحديث الكمية إذا كان العنصر موجودًا
            $cartItem->update([
                'quantity' => $cartItem->quantity + $validated['quantity'],
            ]);
        } else {
            // إضافة عنصر جديد إلى السلة
            $cartItem = $cart->cart_items()->create([
                'product_id' => $validated['product_id'],
                'quantity' => $validated['quantity'],
            ]);
        }

        // تحديث السعر الإجمالي للسلة
        $this->updateCartTotal($cart);

        return response()->json([
            'message' => 'تم إضافة العنصر إلى السلة بنجاح',
            'cart_item' => $cartItem->load('product'),
        ], 201);
    }

    /**
     * إزالة عنصر من السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function removeFromCart(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        // البحث عن سلة المستخدم
        $cart = $user->cart()->first();

        if (!$cart) {
            return response()->json(['message' => 'السلة غير موجودة'], 404);
        }

        // البحث عن العنصر في السلة
        $cartItem = $cart->cart_items()->where('product_id', $validated['product_id'])->first();

        if (!$cartItem) {
            return response()->json(['message' => 'العنصر غير موجود في السلة'], 404);
        }

        // حذف العنصر
        $cartItem->delete();

        // تحديث السعر الإجمالي للسلة
        $this->updateCartTotal($cart);

        return response()->json(['message' => 'تم إزالة العنصر من السلة بنجاح']);
    }

    /**
     * جلب جميع عناصر السلة
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCart(Request $request)
    {
        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        // البحث عن سلة المستخدم
        $cart = $user->cart()->with('cart_items.product')->first();

        if (!$cart) {
            return response()->json([
                'message' => 'السلة فارغة',
                'cart' => null,
            ]);
        }

        return response()->json([
            'message' => 'تم جلب السلة بنجاح',
            'cart' => $cart,
        ]);
    }

    /**
     * تحديث السعر الإجمالي للسلة
     *
     * @param Cart $cart
     * @return void
     */
    protected function updateCartTotal(Cart $cart)
    {
        $total = $cart->cart_items()->with('product')->get()->sum(function ($item) {
            return $item->quantity * ($item->product->price ?? 0);
        });

        $cart->update(['total_price' => $total]);
    }

}