<?php

/**
 * Created by Mohamed.
 */


namespace App\Http\Controllers\API;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Password;
use Laravel\Socialite\Facades\Socialite;

class AuthController extends Controller
{
    public function register(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validated = $request->validate([
            'full_name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
            'password_confirmation' => 'required|string',
        ]);

        // إنشاء مستخدم جديد
        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = User::create([
            'full_name' => $validated['full_name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
        ]);

        // إرسال إشعار تفعيل البريد
        $user->sendEmailVerificationNotification();

        // إصدار توكن Sanctum
        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'message' => 'تم التسجيل بنجاح، يرجى تفعيل بريدك الإلكتروني',
            'user' => $user,
            'token' => $token,
        ], 201);
    }

    // تسجيل الدخول
    public function login(Request $request)
    {
        // التحقق من البيانات المدخلة
        $validated = $request->validate([
            'email' => 'required|string|email',
            'password' => 'required|string',
        ]);

        // محاولة تسجيل الدخول
        if (!Auth::attempt($validated)) {
            return response()->json([
                'message' => 'بيانات تسجيل الدخول غير صحيحة',
            ], 401);
        }

        // الحصول على المستخدم
        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = Auth::user();

        // التحقق من تفعيل البريد
        if (!$user->hasVerifiedEmail()) {
            return response()->json([
                'message' => 'يرجى تفعيل بريدك الإلكتروني أولاً',
            ], 403);
        }

        // إصدار توكن Sanctum
        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'message' => 'تم تسجيل الدخول بنجاح',
            'user' => $user,
            'token' => $token,
        ]);
    }

    // نسيت كلمة المرور
    public function forgotPassword(Request $request)
    {
        // التحقق من البريد الإلكتروني
        $validated = $request->validate([
            'email' => 'required|string|email',
        ]);

        // البحث عن المستخدم
        /** @var \App\Models\User|null $user */
        $user = User::where('email', $validated['email'])->first();

        if (!$user) {
            return response()->json([
                'message' => 'البريد الإلكتروني غير موجود',
            ], 404);
        }

        // التحقق من تفعيل البريد
        if (!$user->hasVerifiedEmail()) {
            return response()->json([
                'message' => 'يرجى تفعيل بريدك الإلكتروني أولاً',
            ], 403);
        }

        // إرسال رابط إعادة تعيين كلمة المرور
        $status = Password::sendResetLink(
            $request->only('email')
        );

        return $status === Password::RESET_LINK_SENT
            ? response()->json(['message' => 'تم إرسال رابط إعادة تعيين كلمة المرور إلى بريدك'])
            : response()->json(['message' => 'فشل إرسال رابط إعادة تعيين كلمة المرور'], 400);
    }

    // إعادة توجيه إلى Google
    public function redirectToGoogle()
    {
        /** @var \Laravel\Socialite\Contracts\Provider|\Laravel\Socialite\Two\GoogleProvider $driver */
        $driver = Socialite::driver('google');
        return $driver->stateless()->redirect();
    }

    // معالجة Callback من Google
    public function handleGoogleCallback()
    {
        try {
            /** @var \Laravel\Socialite\Contracts\Provider|\Laravel\Socialite\Two\GoogleProvider $driver */
            $driver = Socialite::driver('google');
            $socialUser = $driver->stateless()->user();

            // البحث عن المستخدم أو إنشاء مستخدم جديد
            /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
            $user = User::updateOrCreate(
                ['google_id' => $socialUser->getId()],
                [
                    'full_name' => $socialUser->getName(),
                    'email' => $socialUser->getEmail(),
                    'email_verified_at' => now(),
                ]
            );

            $token = $user->createToken('auth_token')->plainTextToken;

            return response()->json([
                'message' => 'تم تسجيل الدخول عبر Google بنجاح',
                'user' => $user,
                'token' => $token,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'فشل تسجيل الدخول عبر Google',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    // إعادة توجيه إلى Facebook
    public function redirectToFacebook()
    {
        /** @var \Laravel\Socialite\Contracts\Provider|\Laravel\Socialite\Two\FacebookProvider $driver */
        $driver = Socialite::driver('facebook');
        return $driver->stateless()->redirect();
    }

    // معالجة Callback من Facebook
    public function handleFacebookCallback()
    {
        try {
            /** @var \Laravel\Socialite\Contracts\Provider|\Laravel\Socialite\Two\FacebookProvider $driver */
            $driver = Socialite::driver('facebook');
            $socialUser = $driver->stateless()->user();

            // البحث عن المستخدم أو إنشاء مستخدم جديد
            /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
            $user = User::updateOrCreate(
                ['facebook_id' => $socialUser->getId()],
                [
                    'full_name' => $socialUser->getName(),
                    'email' => $socialUser->getEmail(),
                    'email_verified_at' => now(),
                ]
            );

            $token = $user->createToken('auth_token')->plainTextToken;

            return response()->json([
                'message' => 'تم تسجيل الدخول عبر Facebook بنجاح',
                'user' => $user,
                'token' => $token,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'فشل تسجيل الدخول عبر Facebook',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    // تسجيل الخروج
    public function logout(Request $request)
    {
        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();
        $user->currentAccessToken()->delete();

        return response()->json(['message' => 'تم تسجيل الخروج بنجاح']);
    }

    // تفعيل البريد الإلكتروني يدويًا (اختياري)
    public function verifyEmail(Request $request)
    {
        /** @var \App\Models\User|\Laravel\Sanctum\HasApiTokens $user */
        $user = $request->user();

        if (!$user) {
            return response()->json(['message' => 'غير مصرح'], 401);
        }

        if ($user->hasVerifiedEmail()) {
            return response()->json(['message' => 'البريد الإلكتروني مفعل مسبقًا']);
        }

        $user->markEmailAsVerified();

        return response()->json(['message' => 'تم تفعيل البريد الإلكتروني بنجاح']);
    }
}